import { TaskDtoSource } from '../../../core/models/jkc-object/jkc-object';
import { GlobalFunctions } from "../../../shared/global-functions";
import { ContactTypeOption, ContextMenuView, OperationType } from "../../enums/common-enums";
import { EntityFqdn, TaskSource } from "../../enums/core-enums";
import { CustomContextMenuContentDto } from "../context-menu/custom-context-menu-contentDto";
export class ContextMenu {
    constructor(oldContentType, contentType, translateService, loadedItem, loggedInUserId, contextMenuView = ContextMenuView.Detail, customContextMenuItems = [], sessionManager, integrationSettingManager, socialMediaService) {
        this.menuItems = [];
        this.contentType = "";
        this.oldContentType = "";
        this.translateService = null;
        this.loadedItem = null;
        this.loggedInUserId = "";
        this.customContextMenuItems = [];
        this.sessionManager = null;
        this.integrationSettingManager = null;
        this.socialMediaService = null;
        this.contentType = contentType;
        this.oldContentType = oldContentType;
        this.translateService = translateService;
        this.loadedItem = loadedItem;
        this.loggedInUserId = loggedInUserId;
        this.customContextMenuItems = customContextMenuItems;
        this.sessionManager = sessionManager;
        this.integrationSettingManager = integrationSettingManager;
        this.socialMediaService = socialMediaService;
        this.updateContextMenuOptions(contextMenuView);
    }
    updateContextMenuOptions(contextMenuView) {
        switch (contextMenuView) {
            case ContextMenuView.Detail:
                {
                    this.loadDetailMenuItem();
                    this.updateCustomContextMenuOptions();
                    break;
                }
            case ContextMenuView.SearchList:
                {
                    this.loadSearchResultMenuItem();
                    break;
                }
        }
    }
    updateCustomContextMenuOptions() {
        if (this.customContextMenuItems?.length > 0) {
            this.customContextMenuItems.forEach(menuItem => {
                if (menuItem.ViewType == 0 && !GlobalFunctions.isEmpty(menuItem.Caption)) {
                    if (this.contentType === GlobalFunctions.getObjectTypeName(menuItem.ContentType)) {
                        const contextMenuItem = new ContextMenuItem(menuItem.Caption, "CUSTOMCONTEXTMENU", menuItem.ContentType);
                        contextMenuItem.customItem = menuItem;
                        this.menuItems.push(contextMenuItem);
                    }
                }
            });
        }
    }
    addMenuItem(title, key, operation, oldContentType, dontCheckRight = false, objectSource = null) {
        if (operation == OperationType.Add && !dontCheckRight) {
            if (GlobalFunctions.hasRight(oldContentType, operation, objectSource) && GlobalFunctions.hasRight(oldContentType, OperationType.View, objectSource))
                this.menuItems.push(new ContextMenuItem(this.translateService.instant(title), key, this.contentType));
        }
        else if (dontCheckRight || GlobalFunctions.hasRight(oldContentType, operation, objectSource))
            this.menuItems.push(new ContextMenuItem(this.translateService.instant(title), key, this.contentType));
    }
    loadSearchResultMenuItem() {
        switch (this.contentType.toUpperCase()) {
            case "TASK":
                {
                    let menuContentType = GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Tasks_Entities_Task]);
                    this.addMenuItem("LBL_SEND_REMINDERS", "SENDREMINDERS", OperationType.SendTaskReminder, menuContentType);
                    if (GlobalFunctions.hasRight(menuContentType, OperationType.CanBulkHandleAnyTask) || GlobalFunctions.hasRight(menuContentType, OperationType.CanBulkHandleTask))
                        this.addMenuItem("LBL_BULK_HANDLE", "BULKHANDLE", OperationType.CanBulkHandleTask, menuContentType, true);
                    this.addMenuItem("LBL_SEND_TO_EXCEL", "SENDTOEXCEL", OperationType.View, menuContentType, true);
                    this.addMenuItem("LBL_PRINT", "SENDTOPRINTER", OperationType.View, menuContentType, true);
                    break;
                }
            case "CONTACTHISTORY":
                {
                    const menuContentType = GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_ContactHistory_Entities_ContactMoment]);
                    this.addMenuItem("LBL_DELETE", "DELETE", OperationType.BulkDelete, menuContentType);
                    this.addMenuItem("LBL_MARK_AS_NEVER_DELETE", "NEVERDELETE", OperationType.BulkDelete, menuContentType);
                    break;
                }
            case "CONTACTMOMENT":
                {
                    let menuContentType = GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_ContactHistory_Entities_ContactMoment]);
                    this.addMenuItem("LBL_SEND_TO_EXCEL", "SENDTOEXCEL", OperationType.View, menuContentType, true);
                    this.addMenuItem("LBL_PRINT", "SENDTOPRINTER", OperationType.View, menuContentType, true);
                    this.addMenuItem("LBL_BULK_DELETE", "BULKDELETE", OperationType.BulkDelete, menuContentType);
                    break;
                }
            case "NOTEENTITY":
                {
                    let menuContentType = GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Notes_Entities_NoteEntity]);
                    this.addMenuItem("LBL_BULK_DELETE", "BULKDELETE", OperationType.BulkDelete, menuContentType);
                    break;
                }
            case "SOCIALMEDIA":
                {
                    let menuContentType = GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_SocialMedia_Entities_SocialMedia]);
                    this.addMenuItem("LBL_BULK_REPLY", "BULKREPLY", OperationType.BulkReplySocialMessage, menuContentType);
                    if (!GlobalFunctions.isEmpty(this.socialMediaService) && !GlobalFunctions.isEmpty(this.socialMediaService?.lastSelectedOption) && this.socialMediaService?.lastSelectedOption.toUpperCase() == "NEW")
                        this.addMenuItem("LBL_BULK_DELETE", "BULKDELETE", OperationType.BulkDelete, menuContentType);
                    if (!GlobalFunctions.isEmpty(this.socialMediaService) && !GlobalFunctions.isEmpty(this.socialMediaService?.lastSelectedOption) && this.socialMediaService?.lastSelectedOption.toUpperCase() == "PUBLISHED")
                        this.addMenuItem("LBL_ARCHIVE", "ARCHIVESOCIALMEDIA", OperationType.ArchiveSocialMedia, menuContentType);
                    break;
                }
            case "TICKET":
                {
                    let menuContentType = GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Tickets_Entities_Ticket]);
                    this.addMenuItem("LBL_SEND_REMINDERS", "SENDREMINDERS", OperationType.SendTicketReminder, menuContentType);
                    this.addMenuItem("LBL_SEND_TO_EXCEL", "SENDTOEXCEL", OperationType.View, menuContentType, true);
                    break;
                }
        }
    }
    loadDetailMenuItem() {
        switch (this.contentType.toUpperCase()) {
            case "CASEENTITY":
                {
                    this.addMenuItem("LBL_ADD_DOCUMENT", "ADDDOCUMENT", OperationType.Add, GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Documents_Entities_Document]));
                    break;
                }
            case "EMAIL":
                {
                    let emailData = this.loadedItem;
                    if (emailData.Header != null) {
                        if (emailData.Header.EmailSource == 'Inbox') {
                            if (GlobalFunctions.hasRight(this.oldContentType, OperationType.CanHandleAnyEmail)) {
                                this.addMenuItem("LBL_REPLY", "REPLY", OperationType.ReplyEmail, this.oldContentType);
                                this.addMenuItem("LBL_FORWARD", "FORWARD", OperationType.ForwardEmail, this.oldContentType);
                            }
                            this.addMenuItem("LBL_ADD_TO_CASE", "ADDTOCASE", OperationType.Add, GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Cases_Entities_CaseEntity]));
                            this.addMenuItem("LBL_ADD_CASE_FROM_EMAIL", "ADDCASEFROMEMAIL", OperationType.Add, GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Cases_Entities_CaseEntity]));
                            this.addMenuItem("LBL_ADD_DOCUMENT_FROM_EMAIL", "ADDDOCUMENTFROMEMAIL", OperationType.Add, GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Documents_Entities_Document]));
                            this.addMenuItemForTicket("EMAIL");
                            if (GlobalFunctions.hasRight(this.oldContentType, OperationType.CanArchiveAnyEmail) || GlobalFunctions.hasRight(this.oldContentType, OperationType.ArchiveEmail))
                                this.addMenuItem("LBL_ARCHIVE", "ARCHIVE", OperationType.ArchiveEmail, this.oldContentType, true);
                        }
                        else {
                            this.addMenuItem("LBL_REPLY", "REPLY", OperationType.ReplyEmail, this.oldContentType);
                            this.addMenuItem("LBL_FORWARD", "FORWARD", OperationType.ForwardEmail, this.oldContentType);
                        }
                    }
                    if (GlobalFunctions.hasRight(this.oldContentType, OperationType.CanDeleteAnyEmail) || GlobalFunctions.hasRight(this.oldContentType, OperationType.Delete))
                        this.addMenuItem("LBL_DELETE", "DELETE", OperationType.Delete, this.oldContentType, true);
                    if (GlobalFunctions.hasRight(GlobalFunctions.toContentType("Task"), OperationType.Add, TaskDtoSource.Assignments) && GlobalFunctions.hasRight(this.oldContentType, OperationType.Reassign))
                        this.addMenuItem("LBL_ADD_ASSIGNMENT", "ADDASSIGNMENT", OperationType.Add, GlobalFunctions.toContentType("Task"), false, TaskDtoSource.Assignments);
                    break;
                }
            case "PRODUCT":
                {
                    let productDto = this.loadedItem;
                    if (this.sessionManager.isSystemAIFeatureEnabled && GlobalFunctions.hasRight(this.oldContentType, OperationType.Summarize))
                        this.addMenuItem("LBL_SUMMARIZE", "SUMMARIZE", OperationType.Summarize, this.oldContentType);
                    if (productDto != null && productDto.Links != null && productDto.Links.length > 0 && productDto.Links[0].UriText != null && productDto.Links[0].UriText != '')
                        this.addMenuItem("LBL_E_FORM", "EFORM", null, this.oldContentType, true);
                    this.addMenuItem("LBL_ADD_CASE", "ADDCASE", OperationType.Add, GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Cases_Entities_CaseEntity]));
                    break;
                }
            case "APPOINTMENT":
                {
                    let taskDto = this.loadedItem;
                    if (!taskDto.IsClosed) {
                        let objectSource = GlobalFunctions.getObjectSourceForTask(this.contentType);
                        let menuContentType = GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Tasks_Entities_Task]);
                        if (taskDto != null && !GlobalFunctions.isEmpty(taskDto.TaskSource) && taskDto.TaskSource === TaskSource.SmartCJM) {
                            if (GlobalFunctions.hasRight(menuContentType, OperationType.CanDeleteAnyTask, objectSource) || GlobalFunctions.hasRight(menuContentType, OperationType.Delete, objectSource))
                                this.addMenuItem("LBL_DELETE", "DELETE", OperationType.Delete, menuContentType);
                            if (GlobalFunctions.hasRight(menuContentType, OperationType.CanEditAnyTask, objectSource) || GlobalFunctions.hasRight(menuContentType, OperationType.Edit, objectSource))
                                this.addMenuItem("LBL_EDIT", "EDIT", OperationType.Edit, menuContentType);
                        }
                        else if (taskDto != null && !GlobalFunctions.isEmpty(taskDto.TaskSource) && taskDto.TaskSource === TaskSource.Qmatic) {
                            if (GlobalFunctions.hasRight(menuContentType, OperationType.CanDeleteAnyTask, objectSource) || (GlobalFunctions.isOwnerOfTask(taskDto) && GlobalFunctions.hasRight(menuContentType, OperationType.Delete, objectSource)))
                                this.addMenuItem("LBL_DELETE", "DELETE", OperationType.Delete, menuContentType);
                        }
                        else if (taskDto != null && !GlobalFunctions.isEmpty(taskDto.TaskSource) && taskDto.TaskSource === TaskSource.Jcc) {
                            if (GlobalFunctions.hasRight(menuContentType, OperationType.CanDeleteAnyTask, objectSource) || GlobalFunctions.hasRight(menuContentType, OperationType.Delete, objectSource))
                                this.addMenuItem("LBL_DELETE", "DELETE", OperationType.Delete, menuContentType);
                        }
                        else {
                            if (taskDto.IsMyTask)
                                this.addMenuItem("LBL_HANDLE", "HANDLE", OperationType.CanHandleAnyTask, menuContentType, true);
                            if (GlobalFunctions.hasRight(menuContentType, OperationType.Reassign, objectSource))
                                this.addMenuItem("LBL_REASSIGN", "REASSIGN", OperationType.Reassign, menuContentType);
                            if (GlobalFunctions.hasRight(menuContentType, OperationType.CanDeleteAnyTask, objectSource) || (GlobalFunctions.isOwnerOfTask(taskDto) && GlobalFunctions.hasRight(menuContentType, OperationType.Delete, objectSource)))
                                this.addMenuItem("LBL_DELETE", "DELETE", OperationType.Delete, menuContentType);
                            if (GlobalFunctions.hasRight(menuContentType, OperationType.CanEditAnyTask, objectSource) || (GlobalFunctions.isOwnerOfTask(taskDto) && GlobalFunctions.hasRight(menuContentType, OperationType.Edit, objectSource)))
                                this.addMenuItem("LBL_EDIT", "EDIT", OperationType.Edit, menuContentType);
                        }
                    }
                    break;
                }
            case "ASSIGNMENT":
            case "INFORMATIONREQUEST":
            case "REPORTIMPROVEMENT":
            case "CALLBACKNOTE":
                {
                    let objectSource = GlobalFunctions.getObjectSourceForTask(this.contentType);
                    let taskDto = this.loadedItem;
                    if (!taskDto.IsClosed) {
                        let menuContentType = GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Tasks_Entities_Task]);
                        let manageGroupcondition = (!GlobalFunctions.hasRight(menuContentType, OperationType.CanManageLocalGroup, objectSource)
                            || (GlobalFunctions.hasRight(menuContentType, OperationType.CanManageLocalGroup, objectSource) && !taskDto.HasPartialViewRights));
                        if (taskDto.IsMyTask)
                            this.addMenuItem("LBL_HANDLE", "HANDLE", OperationType.CanHandleAnyTask, menuContentType, true);
                        if (manageGroupcondition && GlobalFunctions.hasRight(menuContentType, OperationType.Reassign, objectSource))
                            this.addMenuItem("LBL_REASSIGN", "REASSIGN", OperationType.Reassign, menuContentType);
                        if (manageGroupcondition && (GlobalFunctions.hasRight(menuContentType, OperationType.CanDeleteAnyTask, objectSource) || (GlobalFunctions.isOwnerOfTask(taskDto) && GlobalFunctions.hasRight(menuContentType, OperationType.Delete, objectSource))))
                            this.addMenuItem("LBL_DELETE", "DELETE", OperationType.Delete, GlobalFunctions.toContentType("TASK"));
                        if (manageGroupcondition && ((GlobalFunctions.isOwnerOfTask(taskDto) && GlobalFunctions.hasRight(menuContentType, OperationType.Edit, objectSource)) || GlobalFunctions.hasRight(menuContentType, OperationType.CanEditAnyTask, objectSource)))
                            this.addMenuItem("LBL_EDIT", "EDIT", OperationType.Edit, menuContentType);
                        let sendToTaskCaption = this.integrationSettingManager.getIntegrationSetting(EntityFqdn.Decos_Kcs_Tasks_Entities_Task, "SENDTOTASKCAPTION");
                        if (!GlobalFunctions.isEmpty(sendToTaskCaption))
                            this.menuItems.push(new ContextMenuItem(sendToTaskCaption, "SENDTOTASK", this.contentType));
                    }
                    break;
                }
            case "DOCUMENT":
                {
                    this.addMenuItem("LBL_ADD_DOCUMENT", "ADD", OperationType.Add, this.oldContentType);
                    this.addMenuItemForTicket("DOCUMENT");
                    this.addMenuItem("LBL_ADD_ASSIGNMENT", "ADDASSIGNMENT", OperationType.Add, GlobalFunctions.toContentType("Task"), false, TaskDtoSource.Assignments);
                    break;
                }
            case "EMPLOYEE":
                {
                    this.addMenuItem("LBL_CALLBACK_NOTE", "CALLBACKNOTE", OperationType.Add, GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Tasks_Entities_Task]), false, TaskDtoSource.CallbackNotes);
                    if (GlobalFunctions.hasRight(this.oldContentType, OperationType.CanEditAnyEmployee) || (GlobalFunctions.hasRight(this.oldContentType, OperationType.Edit) && this.loadedItem.Id === GlobalFunctions.getLoggedInUserId()))
                        this.addMenuItem("LBL_EDIT", "EDIT", OperationType.Edit, this.oldContentType);
                    break;
                }
            case "NOTEENTITY":
                {
                    if ((GlobalFunctions.hasRight(this.oldContentType, OperationType.CanEditAnyNote) || GlobalFunctions.hasRight(this.oldContentType, OperationType.Edit)) && !this.loadedItem.IsPrivate)
                        this.addMenuItem("LBL_EDIT", "EDIT", OperationType.Edit, this.oldContentType, true);
                    if ((GlobalFunctions.hasRight(this.oldContentType, OperationType.CanDeleteAnyNote) || GlobalFunctions.hasRight(this.oldContentType, OperationType.Delete)) && !this.loadedItem.IsPrivate)
                        this.addMenuItem("LBL_DELETE", "DELETE", OperationType.Delete, this.oldContentType);
                    break;
                }
            case "FILEENTITY":
                {
                    if (GlobalFunctions.hasRight(this.oldContentType, OperationType.Edit))
                        this.addMenuItem("LBL_EDIT", "EDIT", OperationType.Edit, this.oldContentType);
                    if (GlobalFunctions.hasRight(this.oldContentType, OperationType.Delete))
                        this.addMenuItem("LBL_DELETE", "DELETE", OperationType.Delete, this.oldContentType);
                    break;
                }
            case "LINK":
                {
                    if (GlobalFunctions.hasRight(this.oldContentType, OperationType.Edit))
                        this.addMenuItem("LBL_EDIT", "EDIT", OperationType.Edit, this.oldContentType);
                    if (GlobalFunctions.hasRight(this.oldContentType, OperationType.Delete))
                        this.addMenuItem("LBL_DELETE", "DELETE", OperationType.Delete, this.oldContentType);
                    break;
                }
            case "CONTACTMOMENT":
                {
                    let manageGroupcondition = (!GlobalFunctions.hasRight(this.oldContentType, OperationType.CanManageLocalGroup) || (GlobalFunctions.hasRight(this.oldContentType, OperationType.CanManageLocalGroup) && !this.loadedItem.HasPartialViewRights));
                    if (manageGroupcondition && GlobalFunctions.hasRight(this.oldContentType, OperationType.Edit)) {
                        this.addMenuItem("LBL_EDIT", "EDIT", OperationType.Edit, this.oldContentType);
                        if (this.loadedItem.NeverDelete)
                            this.addMenuItem("LBL_MARK_AS_DELETABLE", "MARKASDELETABLE", OperationType.Edit, this.oldContentType);
                        else
                            this.addMenuItem("LBL_MARK_AS_NEVER_DELETE", "MARKASNEVERDELETE", OperationType.Edit, this.oldContentType);
                    }
                    if (manageGroupcondition && GlobalFunctions.hasRight(this.oldContentType, OperationType.Delete))
                        this.addMenuItem("LBL_DELETE", "DELETE", OperationType.Delete, this.oldContentType);
                    break;
                }
            case "PERSON":
                {
                    if (GlobalFunctions.hasRight(this.oldContentType, OperationType.CanEditAnyNote) || GlobalFunctions.hasRight(this.oldContentType, OperationType.Edit))
                        this.addMenuItem("LBL_EDIT", "EDIT", OperationType.Edit, this.oldContentType);
                    this.addMenuItemForTicket("PERSON");
                    if (GlobalFunctions.hasRight(this.oldContentType, OperationType.MergeLetterTemplatePerson))
                        this.addMenuItem("LBL_MERGE_LETTER_TEMPLATE", "MERGELETTERTEMPLATE", OperationType.MergeLetterTemplatePerson, this.oldContentType);
                    if (!GlobalFunctions.isEmpty(this.loadedItem) && !GlobalFunctions.isEmpty(this.loadedItem.ObjectSource) && this.loadedItem.ObjectSource.toLowerCase().indexOf('.cdr') != -1) {
                        if (GlobalFunctions.hasRight(this.oldContentType, OperationType.Delete))
                            this.addMenuItem("LBL_DELETE", "DELETE", OperationType.Delete, this.oldContentType);
                        if (GlobalFunctions.hasRight(this.oldContentType, OperationType.MergeData))
                            this.addMenuItem("LBL_MERGEDATA", "MERGEDATA", OperationType.MergeData, this.oldContentType, true);
                    }
                    break;
                }
            case "ORGANIZATION":
                {
                    if (GlobalFunctions.hasRight(this.oldContentType, OperationType.Edit))
                        this.addMenuItem("LBL_EDIT", "EDIT", OperationType.Edit, this.oldContentType);
                    let menuContentType = GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Persons_Entities_Person]);
                    if (GlobalFunctions.hasRight(menuContentType, OperationType.Add))
                        this.addMenuItem("LBL_ADD_REPRESENTATIVE", "ADDREPRESENTATIVE", OperationType.Add, menuContentType);
                    this.addMenuItemForTicket("ORGANIZATION");
                    if (GlobalFunctions.hasRight(this.oldContentType, OperationType.MergeLetterTemplateOrganization))
                        this.addMenuItem("LBL_MERGE_LETTER_TEMPLATE", "MERGELETTERTEMPLATE", OperationType.MergeLetterTemplateOrganization, this.oldContentType);
                    if (!GlobalFunctions.isEmpty(this.loadedItem.ObjectSource) && this.loadedItem.ObjectSource.toLowerCase().indexOf('.cdr') != -1) {
                        if (GlobalFunctions.hasRight(this.oldContentType, OperationType.Delete))
                            this.addMenuItem("LBL_DELETE", "DELETE", OperationType.Delete, this.oldContentType);
                        if (GlobalFunctions.hasRight(this.oldContentType, OperationType.MergeData))
                            this.addMenuItem("LBL_MERGEDATA", "MERGEDATA", OperationType.MergeData, this.oldContentType, true);
                    }
                    break;
                }
            case "THEME":
                {
                    if (this.sessionManager.isSystemAIFeatureEnabled && GlobalFunctions.hasRight(this.oldContentType, OperationType.Summarize))
                        this.addMenuItem("LBL_SUMMARIZE", "SUMMARIZE", OperationType.Summarize, this.oldContentType);
                    if (GlobalFunctions.hasRight(this.oldContentType, OperationType.Edit) && !this.loadedItem.IsPrivate)
                        this.addMenuItem("LBL_EDIT", "EDIT", OperationType.Edit, this.oldContentType);
                    if (GlobalFunctions.hasRight(this.oldContentType, OperationType.Delete) && !this.loadedItem.IsPrivate)
                        this.addMenuItem("LBL_DELETE", "DELETE", OperationType.Delete, this.oldContentType);
                    break;
                }
            case "WHATSAPP":
            case "TWITTER":
            case "FACEBOOK":
                {
                    if (this.loadedItem.IsPublishedMessage == true)
                        break;
                    let menuContentType = GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_SocialMedia_Entities_SocialMedia]);
                    this.addMenuItem("LBL_CALLBACK_NOTE", "CALLBACKNOTE", OperationType.Add, GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Tasks_Entities_Task]), false, TaskDtoSource.CallbackNotes);
                    this.addMenuItem("LBL_INFORMATION_REQUEST", "INFORMATIONREQUEST", OperationType.Add, GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Tasks_Entities_Task]), false, TaskDtoSource.InformationRequest);
                    if (!GlobalFunctions.isEmpty(this.loadedItem) && !GlobalFunctions.isEmpty(this.loadedItem.MessageHandling) && !GlobalFunctions.isEmpty(this.loadedItem.MessageHandling.HandlerId))
                        this.addMenuItem("LBL_REASSIGN", "REASSIGN", OperationType.Reassign, menuContentType);
                    if (!GlobalFunctions.isEmpty(this.loadedItem) && GlobalFunctions.isEmpty(this.loadedItem.MessageHandling))
                        this.addMenuItem("LBL_DELETE", "DELETE", OperationType.Delete, menuContentType);
                    this.addMenuItemForTicket("SOCIALMEDIA");
                    if (GlobalFunctions.hasRight(GlobalFunctions.toContentType("Task"), OperationType.Add, TaskDtoSource.Assignments) && (!GlobalFunctions.isEmpty(this.socialMediaService?.lastSelectedOption) && this.socialMediaService?.lastSelectedOption.toUpperCase() == "NEW"))
                        this.addMenuItem("LBL_ADD_ASSIGNMENT", "ADDASSIGNMENT", OperationType.Add, GlobalFunctions.toContentType("Task"), false, TaskDtoSource.Assignments);
                    break;
                }
            case "INFORMATIONRESPONSE":
                {
                    this.addMenuItem("LBL_ARCHIVE", "ARCHIVE", OperationType.Edit, GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Tasks_Entities_Task]), false, "Decos.Kcs.InformationRequests.Cdr");
                    break;
                }
            case "TICKET":
                {
                    let menuContentType = GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Tickets_Entities_Ticket]);
                    if ((this.loadedItem != null && !GlobalFunctions.isEmpty(this.loadedItem.TicketTypeId) && GlobalFunctions.hasRight(menuContentType, OperationType.Edit, this.loadedItem.TicketTypeId)) || GlobalFunctions.hasRight(menuContentType, OperationType.CanEditAnyTicket)) {
                        if (this.loadedItem.State && this.loadedItem.State.toLowerCase() != "afgesloten" || GlobalFunctions.hasRight(menuContentType, OperationType.CanEditClosedTicket, this.loadedItem.TicketTypeId))
                            this.addMenuItem("LBL_EDIT", "EDIT", OperationType.Edit, menuContentType);
                    }
                    if (this.loadedItem.IsMyTicket && this.loadedItem.State && this.loadedItem.State.toLowerCase() != "afgesloten")
                        this.addMenuItem("LBL_HANDLE", "HANDLE", OperationType.CanHandleAnyTicket, menuContentType, true);
                    this.addMenuItem("LBL_INFORMATION_REQUEST", "INFORMATIONREQUEST", OperationType.Add, GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Tasks_Entities_Task]), false, TaskDtoSource.InformationRequest);
                    if (GlobalFunctions.hasRight(menuContentType, OperationType.Reassign))
                        this.addMenuItem("LBL_REASSIGN", "REASSIGN", OperationType.Reassign, menuContentType);
                    this.addMenuItem("LBL_ADD_DOCUMENT", "ADDDOCUMENT", OperationType.Add, GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Documents_Entities_Document]));
                    if (this.loadedItem != null && !GlobalFunctions.isEmpty(this.loadedItem.TicketTypeId) && GlobalFunctions.hasRight(this.oldContentType, OperationType.Delete, this.loadedItem.TicketTypeId))
                        this.addMenuItem("LBL_DELETE", "DELETE", OperationType.Delete, menuContentType);
                    if (GlobalFunctions.hasRight(GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Tickets_Entities_Ticket]), OperationType.Link))
                        this.addMenuItem("LBL_LINK_FROM_SESSION", "LINKFROMSESSION", OperationType.Link, GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Tickets_Entities_Ticket]));
                    break;
                }
        }
        if (this.contentType.toLowerCase() != "informationresponse" && this.contentType.toLowerCase() != "informationrequest" && this.contentType.toLowerCase() != "email" && this.contentType.toLowerCase() != "whatsapp" && this.contentType.toLowerCase() != "twitter" && this.contentType.toLowerCase() != "facebook" && (GlobalFunctions.isEmpty(this.loadedItem?.TaskSource) || this.loadedItem?.TaskSource != TaskSource.Jcc)) {
            if (this.contentType.toLowerCase() != "fileentity" && this.contentType.toLowerCase() != "link") {
                let noteContentType = GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Notes_Entities_NoteEntity]);
                if (this.contentType.toLowerCase() != "ticket" && this.contentType.toLowerCase() != "noteentity" && !GlobalFunctions.isTaskDomain(this.contentType.toUpperCase()) || (GlobalFunctions.isTaskDomain(this.contentType.toUpperCase()) && this.loadedItem != null && this.loadedItem.IsClosed == false))
                    this.addMenuItem("LBL_ADD_IMPROVEMENT", "ADDIMPROVEMENT", OperationType.Add, GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Tasks_Entities_Task]), false, TaskDtoSource.ReportImprovements);
                if (this.contentType.toLowerCase() != "ticket" && this.contentType.toLowerCase() != "theme" && !GlobalFunctions.isTaskDomain(this.contentType.toUpperCase()) || (GlobalFunctions.isTaskDomain(this.contentType.toUpperCase()) && this.loadedItem != null && this.loadedItem.IsClosed == false))
                    this.addMenuItem("LBL_ADD_NOTE", "ADDNOTE", OperationType.Add, noteContentType);
            }
            let objectSource = null;
            if (GlobalFunctions.isTaskDomain(this.contentType.toUpperCase())) {
                objectSource = GlobalFunctions.getObjectSourceForTask(this.contentType);
                this.oldContentType = GlobalFunctions.toContentType("TASK");
            }
            else if (this.contentType.toUpperCase() == "DOMAIN" && this.loadedItem != null)
                objectSource = this.loadedItem.ObjectSource;
            if (!GlobalFunctions.isTaskDomain(this.contentType.toUpperCase())) {
                this.addMenuItem("LBL_LINK", "LINK", OperationType.Link, this.oldContentType, objectSource);
                this.addMenuItem("LBL_UNLINK", "UNLINK", OperationType.Unlink, this.oldContentType, objectSource);
            }
            if (this.contentType.toLowerCase() != "theme" && this.contentType.toLowerCase() != "link" && this.contentType.toLowerCase() != "issue") {
                if (GlobalFunctions.hasRight(this.oldContentType, OperationType.Print, objectSource) || GlobalFunctions.hasRight(this.oldContentType, OperationType.Email, objectSource))
                    this.addMenuItem("LBL_ADD_TO_BASKET", "ADDTOBASKET", OperationType.Email, this.oldContentType, true, objectSource);
                if (this.sessionManager.employeeStatus.hasChannelEnabled(ContactTypeOption.Email))
                    this.addMenuItem("LBL_SEND_VIA_EMAIL", "SENDVIAEMAIL", OperationType.Email, this.oldContentType, false, objectSource);
                this.addMenuItem("LBL_PRINT", "PRINT", OperationType.Print, this.oldContentType, false, objectSource);
            }
        }
        if (this.contentType.toLowerCase() === "product")
            this.addMenuItem("LBL_PRODUCT_ID", "PRODUCTID", null, this.oldContentType, true);
        if (this.loadedItem != null && this.loadedItem.SourceApplicationDeeplink)
            this.addMenuItem("LBL_SOURCE_APPLICATION_DEEP_LINK", "SOURCEAPPLICATIONDEEPLINK", OperationType.Edit, this.oldContentType, true);
    }
    addMenuItemForTicket(entityType) {
        if (this.sessionManager.isTicketFeatureEnabled) {
            if (entityType != "PERSON" && entityType != "ORGANIZATION" && GlobalFunctions.hasRight(GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Tickets_Entities_Ticket]), OperationType.Link)) {
                const menuItemTitle = this.translateService.instant("LBL_LINK_TO_TICKET").replace(/{TICKETTITLE}/g, this.translateService.instant("LBL_TICKET").toLowerCase());
                this.addMenuItem(menuItemTitle, "LINKTOTICKET", OperationType.Link, GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Tickets_Entities_Ticket]));
            }
            const ticketConfiguration = this.integrationSettingManager.getIntegrationSetting(EntityFqdn.Decos_Kcs_Tickets_Entities_Ticket, "TICKETSCONFIGURATION");
            if (ticketConfiguration?.TicketTypes?.length > 0) {
                ticketConfiguration.TicketTypes.forEach(ticketType => {
                    if (ticketType.HasAddRight) {
                        const menuItemTitle = this.translateService.instant("LBL_ADD_TICKET_FOR_TICKET_TYPE").replace(/{TICKETTITLE}/g, this.translateService.instant("LBL_TICKET").toLowerCase()).replace(/{TICKETTYPE}/g, ticketType.Title.toLowerCase());
                        const contextMenuItem = new ContextMenuItem(menuItemTitle, "ADDTICKET", this.contentType);
                        contextMenuItem.customItem = new CustomContextMenuContentDto();
                        contextMenuItem.customItem.StringRepresentation = ticketType.Title;
                        this.menuItems.push(contextMenuItem);
                    }
                });
            }
        }
    }
}
export class ContextMenuItem {
    constructor(title, key, contentType) {
        this.customItem = null;
        this.title = title;
        this.key = key;
        this.contentType = contentType;
    }
}
//# sourceMappingURL=context-menu.js.map