/*
    This script contains extra Openlayers classes to be used in the
    PdokKaart-api. Currently, these are:
    
    OpenLayers.Format.KMLv2_2
      Symbology support for the OpenLayers class for reading/writing KML 2.2

    OpenLayers.Control.GeocoderControl
      The GeocoderControl control adds pdok search functionality.
    
*/

OpenLayers.Format.KMLv2_2 = OpenLayers.Class(OpenLayers.Format.KML, {
    /**
     * APIProperty: kmlns
     * {String} KML Namespace to use. Defaults to 2.0 namespace.
     */
    kmlns: "http://earth.google.com/kml/2.2",
    
    /**
     * 
     * @param {type} features
     * @returns {string} the KML document
     */
    write: function(features) {
        this.styles = {};

        if(!(features instanceof Array)) {
            features = [features];
        }
        var kml = this.createElementNS(this.kmlns, "kml");
        var document = this.createDocumentXML();
        var folder = this.createFolderXML();
        for(var i=0, len=features.length; i<len; ++i) {
            var styleId = this.extractStyles ? 
                this.createStyleNode(document, this.computedStyle(features[i])) :
                false;
            folder.appendChild(this.createPlacemarkXML(features[i], styleId));
        }
        kml.appendChild(folder);
        document.appendChild(folder);
        kml.appendChild(document);
        return OpenLayers.Format.XML.prototype.write.apply(this, [kml]);
    },
        /**
     * Method: createDocumentXML
     * Creates and returns a KML document node
     *
     * Returns:
     * {DOMElement}
     */
    createDocumentXML: function() {
        // Document name
        var documentName = this.createElementNS(this.kmlns, "name");
        var documentNameText = this.createTextNode(this.foldersName);
        documentName.appendChild(documentNameText);

        // Document description
        var documentDesc = this.createElementNS(this.kmlns, "description");
        var documentDescText = this.createTextNode(this.foldersDesc);
        documentDesc.appendChild(documentDescText);

        var document = this.createElementNS(this.kmlns, "Document");
        document.appendChild(documentName);
        document.appendChild(documentDesc);

        return document;
    },
        createPlacemarkXML: function(feature, styleId) {
        // Placemark name
        var placemarkName = this.createElementNS(this.kmlns, "name");
        var name = feature.style && feature.style.label ? feature.style.label :
                   feature.attributes.name || feature.id;
        placemarkName.appendChild(this.createTextNode(name));

        // Placemark description
        var placemarkDesc = this.createElementNS(this.kmlns, "description");
        var desc = feature.attributes.description || this.placemarksDesc;
        placemarkDesc.appendChild(this.createTextNode(desc));
        
        // Placemark
        var placemarkNode = this.createElementNS(this.kmlns, "Placemark");
        if(feature.fid) {
            placemarkNode.setAttribute("id", feature.fid);
        }
        placemarkNode.appendChild(placemarkName);
        placemarkNode.appendChild(placemarkDesc);

        if (styleId) {
            var styleNode = this.createElementNS(this.kmlns, "styleUrl");
            styleNode.appendChild(this.createTextNode(styleId));
            placemarkNode.appendChild(styleNode);
        }

        // Geometry node (Point, LineString, etc. nodes)
        var geometryNode = this.buildGeometryNode(feature.geometry);
        placemarkNode.appendChild(geometryNode);        
        
        // output attributes as extendedData
        if (feature.attributes) {
            var edNode = this.buildExtendedData(feature.attributes);
            if (edNode) {
                placemarkNode.appendChild(edNode);
            }
        }
        
        return placemarkNode;
    },
    computedStyle: function(feature) {
        if (feature.style) {
            return feature.style;
        } else if (feature.layer) {
            if (feature.layer.style) {
                return feature.layer.style;
            } else {
                return feature.layer.styleMap.createSymbolizer(feature);
            }
        }
    },
    createKmlColorNode: function(color, opacity) {
        var alpha = "ff";
        if (opacity) {
            alpha = Math.round(parseFloat(opacity) * 255).toString(16);
        }
        // TBD: handle '#ccc', 'red'
        // only match '#rrggbb'
        var r = color.slice(1, 3);
        var g = color.slice(3, 5);
        var b = color.slice(5, 7);
        var colorNode = this.createElementNS(this.kmlns, "color");
        colorNode.appendChild(this.createTextNode(alpha + b + g + r));
        return colorNode;
    },

    createStyleNode: function(document, style) {
        if (!style){
            return false;
        } else {
            var styleNode = this.createElementNS(this.kmlns, "Style");
            var id = OpenLayers.Util.createUniqueID("style_");
            styleNode.setAttribute("id", id);

            // LineStyle
            if (style.strokeColor) {
                var lineNode = this.createElementNS(this.kmlns, "LineStyle");
                var colorNode = this.createKmlColorNode(style.strokeColor, style.strokeOpacity);
                lineNode.appendChild(colorNode);

                if (style.strokeWidth) {
                    var width = this.createElementNS(this.kmlns, "width");
                    width.appendChild(this.createTextNode(style.strokeWidth));
                    lineNode.appendChild(width);
                }
                styleNode.appendChild(lineNode);
            }

            // PolyStyle
            if (style.fillColor) {
                var polyNode = this.createElementNS(this.kmlns, "PolyStyle");
                var colorNode = this.createKmlColorNode(style.fillColor, style.fillOpacity);
                polyNode.appendChild(colorNode);
                styleNode.appendChild(polyNode);
            } else /*if (style.fillColor == "none")*/ {
                var polyNode = this.createElementNS(this.kmlns, "PolyStyle");
                var fill = this.createElementNS(this.kmlns, "fill");
                fill.appendChild(this.createTextNode("1"));
                var colorNode = this.createKmlColorNode("#ffffff", "0.01");
                polyNode.appendChild(colorNode);
                polyNode.appendChild(fill);
                styleNode.appendChild(polyNode);
            }
            if (polyNode && style.strokeWidth === "0") {
                var outline = this.createElementNS(this.kmlns, "outline");
                outline.appendChild(this.createTextNode("1"));
                polyNode.appendChild(outline);
                styleNode.appendChild(polyNode);
            }

            // IconStyle
            if (style.externalGraphic) {
                var iconstyleNode = this.createElementNS(this.kmlns, "IconStyle");
                var iconNode = this.createElementNS(this.kmlns, "Icon");

                var href = this.createElementNS(this.kmlns, "href");
                var urlObj = OpenLayers.Util.createUrlObject(
                    style.externalGraphic,
                    {ignorePort80: true}
                );
                if(urlObj.port && urlObj.port !== "80"){
                    url = [urlObj.protocol, '//', urlObj.host, ':', urlObj.port, urlObj.pathname].join('');
                } else {
                    url = [urlObj.protocol, '//', urlObj.host, urlObj.pathname].join('');
                }
                href.appendChild(this.createTextNode(url));
                iconNode.appendChild(href);
                iconstyleNode.appendChild(iconNode);
                var scaleNode = this.createElementNS(this.kmlns, "scale");

                // in KML 2.2, w and h <Icon> attributes are deprecated
                // this means that we can't modify the width/height ratio of the image
                var scale = style.graphicWidth || style.graphicHeight || style.pointRadius * 2;
                scaleNode.appendChild(this.createTextNode(scale/32));
                iconstyleNode.appendChild(scaleNode);
                styleNode.appendChild(iconstyleNode);
            }

            // LabelStyle
            if (style.fontColor) {
                var colorNode = this.createKmlColorNode(style.fontColor, style.fontOpacity);
                var labelStyle = this.createElementNS(this.kmlns, "LabelStyle");
                labelStyle.appendChild(colorNode);
                styleNode.appendChild(labelStyle);
            }
            document.appendChild(styleNode);
            return "#" + id;
        }
    }
});

/**
 * @requires OpenLayers/Control.js
 */

/* Berend DWE/8068: removed GeocoderControl, we use our own version */

/* Copyright (c) 2006-2008 MetaCarta, Inc., published under the Clear BSD
 * license.  See http://svn.openlayers.org/trunk/openlayers/license.txt for the
 * full text of the license. */

/**
 * @requires OpenLayers/Control.js
 *
 * Class: OpenLayers.Control.LoadingPanel
 * In some applications, it makes sense to alert the user that something is
 * happening while tiles are loading. This control displays a div across the
 * map when this is going on.
 *
 * Inherits from:
 *  - <OpenLayers.Control>
 */
OpenLayers.Control.LoadingPanel = OpenLayers.Class(OpenLayers.Control, {

    /**
     * Property: counter
     * {Integer} A counter for the number of layers loading
     */
    counter: 0,

    /**
     * Property: maximized
     * {Boolean} A boolean indicating whether or not the control is maximized
    */
    maximized: false,

    /**
     * Property: visible
     * {Boolean} A boolean indicating whether or not the control is visible
    */
    visible: true,

    /**
     * Constructor: OpenLayers.Control.LoadingPanel
     * Display a panel across the map that says 'loading'.
     *
     * Parameters:
     * options - {Object} additional options.
     */
    initialize: function(options) {
         OpenLayers.Control.prototype.initialize.apply(this, [options]);
    },

    /**
     * Function: setVisible
     * Set the visibility of this control
     *
     * Parameters:
     * visible - {Boolean} should the control be visible or not?
    */
    setVisible: function(visible) {
        this.visible = visible;
        if (visible) {
            OpenLayers.Element.show(this.div);
        } else {
            OpenLayers.Element.hide(this.div);
        }
    },

    /**
     * Function: getVisible
     * Get the visibility of this control
     *
     * Returns:
     * {Boolean} the current visibility of this control
    */
    getVisible: function() {
        return this.visible;
    },

    /**
     * APIMethod: hide
     * Hide the loading panel control
    */
    hide: function() {
        this.setVisible(false);
    },

    /**
     * APIMethod: show
     * Show the loading panel control
    */
    show: function() {
        this.setVisible(true);
    },

    /**
     * APIMethod: toggle
     * Toggle the visibility of the loading panel control
    */
    toggle: function() {
        this.setVisible(!this.getVisible());
    },

    /**
     * Method: addLayer
     * Attach event handlers when new layer gets added to the map
     *
     * Parameters:
     * evt - {Event}
    */
    addLayer: function(evt) {
        if (evt.layer) {
            evt.layer.events.register('loadstart', this, this.increaseCounter);
            evt.layer.events.register('loadend', this, this.decreaseCounter);
        }
    },

    /**
     * Method: setMap
     * Set the map property for the control and all handlers.
     *
     * Parameters:
     * map - {<OpenLayers.Map>} The control's map.
     */
    setMap: function(map) {
        OpenLayers.Control.prototype.setMap.apply(this, arguments);
        this.map.events.register('preaddlayer', this, this.addLayer);
        for (var i = 0; i < this.map.layers.length; i++) {
            var layer = this.map.layers[i];
            layer.events.register('loadstart', this, this.increaseCounter);
            layer.events.register('loadend', this, this.decreaseCounter);
        }
    },

    /**
     * Method: increaseCounter
     * Increase the counter and show control
    */
    increaseCounter: function() {
        this.counter++;
        if (this.counter > 0) {
            if (!this.maximized && this.visible) {
                this.maximizeControl();
            }
        }
    },

    /**
     * Method: decreaseCounter
     * Decrease the counter and hide the control if finished
    */
    decreaseCounter: function() {
        if (this.counter > 0) {
            this.counter--;
        }
        if (this.counter == 0) {
            if (this.maximized && this.visible) {
                this.minimizeControl();
            }
        }
    },

    /**
     * Method: draw
     * Create and return the element to be splashed over the map.
     */
    draw: function () {
        OpenLayers.Control.prototype.draw.apply(this, arguments);
        return this.div;
    },

    /**
     * Method: minimizeControl
     * Set the display properties of the control to make it disappear.
     *
     * Parameters:
     * evt - {Event}
     */
    minimizeControl: function(evt) {
        this.div.style.display = "none";
        this.maximized = false;

        if (evt != null) {
            OpenLayers.Event.stop(evt);
        }
    },

    /**
     * Method: maximizeControl
     * Make the control visible.
     *
     * Parameters:
     * evt - {Event}
     */
    maximizeControl: function(evt) {
        this.div.style.display = "block";
        this.maximized = true;

        if (evt != null) {
            OpenLayers.Event.stop(evt);
        }
    },

    /**
     * Method: destroy
     * Destroy control.
     */
    destroy: function() {
        if (this.map) {
            this.map.events.unregister('preaddlayer', this, this.addLayer);
            if (this.map.layers) {
                for (var i = 0; i < this.map.layers.length; i++) {
                    var layer = this.map.layers[i];
                    layer.events.unregister('loadstart', this,
                        this.increaseCounter);
                    layer.events.unregister('loadend', this,
                        this.decreaseCounter);
                }
            }
        }
        OpenLayers.Control.prototype.destroy.apply(this, arguments);
    },

    CLASS_NAME: "OpenLayers.Control.LoadingPanel"

});