/**
 * @requires OpenLayers/Control.js
 */

/**
 * Class: OpenLayers.Control.Attribution
 * The attribution control adds attribution from layers to the map display. 
 * It uses 'attribution' property of each layer.
 *
 * Inherits from:
 *  - <OpenLayers.Control>
 */
OpenLayers.Control.GeocoderControl =
    OpenLayers.Class(OpenLayers.Control, {

        // PDOK
        geocoderUrl: 'https://nominatim.openstreetmap.org/search?format=xml&polygon=0&limit=1&addressdetails=1',
        geocoderParameter: 'q',

        zoomScale: {
            adres: 13,
            postcode: 11,
            plaats: 8,
            gemeente: 8,
            provincie: 5,
            standaard: 9
        },
        resultFeatures: null, // Berend DWE/8068: parsed list of result features
        decosSearchResultsId: null,  // Berend DWE/8068: id of Decos-stored search results
        searchString: "",
        /**
        * Constructor:
        * 
        * Parameters:
        * options - {Object} Options for control.
        */
        initialize: function (options) {
            OpenLayers.Control.prototype.initialize.apply(this, arguments);
            // we create the div ourselves, to be able to put it outside the map-div
            // if we let OpenLayers create it, and let it be part of the map-div
            // then OpenLayers steals the cursor from our input
            if (!this.div) {
                this.div = document.createElement("div");
                this.div.className = "pdokGeocoderDiv";
                this.div.id = this.div.className + '_' + this.id;
            }
            this.proj4326 = null; // Berend DWE/8257: international geocoders return coordinates as latitude/longitude
            this.allowSelection = true;

            // deferred event delegation:
            // http://davidwalsh.name/event-delegate
            // we need some IE specific magic here
            var me = this;
            var clickFunc = function (e) {
                var target;
                // IE8 does not have an event.target
                e.target ? target = e.target : target = e.srcElement;
                if (target && target.className == "closeWindow") {
                    me.hideResults();
                }
                //else if (target && target.nodeName == "A") {
                //  var a = document.getElementById(target.id);
                //  if (a) {
                //    var x = a.attributes['x'].value;
                //    var y = a.attributes['y'].value;
                //    var z = a.attributes['z'].value;
                //    var i = a.attributes['i'].value;
                //    if (x && y) {
                //      me.map.setCenter(new OpenLayers.LonLat(x, y), z);
                //      me.showResultPoint(i, a.textContent || a.innerText);
                //    }
                //  }
                //  return false;
                //}
            }
            if (this.div.addEventListener) {
                this.div.addEventListener("click", clickFunc);
            } else {
                this.div.attachEvent("onclick", clickFunc); // IE8
            }
        },

        /** 
        * Method: destroy
        * Destroy control.
        */
        destroy: function () {
            OpenLayers.Control.prototype.destroy.apply(this, arguments);
        },

        /**
        * Method: draw
        * Initialize control.
        * 
        * Returns: 
        * {DOMElement} A reference to the DIV DOMElement containing the control
        */
        draw: function () {
            this.map.div.parentNode.insertBefore(this.div, this.map.div);
            OpenLayers.Control.prototype.draw.apply(this, arguments);

            this.div.innerHTML =
                '<form action="#" id="pdoksearchform" name="pdoksearchform" onsubmit="return false;">' +
                '<input id="pdoksearchinput" style="display:none" type="text" value="" name="pdoksearchinput" class="autopopulate default" title="adres, postcode of plaatsnaam" />' +
                '<button type="submit" style="display:none" id="pdoksearchbutton"> Zoek Locatie</button>' +
                '<div id="pdoksearchresults" style="width:320px">&nbsp;</div></form>';
            var me = this;

            $('#pdoksearchresults').delegate('li a', 'click', function (evt) {
                var x = $(this).attr('x');
                var y = $(this).attr('y');
                var z = $(this).attr('z');
                var i = $(this).attr('i');
                if (x && y) {
                    me.map.setCenter(new OpenLayers.LonLat(x, y), z);
                    me.showResultPoint(i, $(this).text());
                }
                else {
                    alert("fout met coordinaten");
                }
                return false;
            });

            document.getElementById('pdoksearchbutton').onclick = function () {
                var searchString = document.getElementById('pdoksearchinput').value;
                me.search(searchString);
                if (document.getElementById('pdoksearchresults').style.display == 'none') {
                    me.showResults();
                }
                else {
                    me.hideResults();
                }
            }
            return this.div;
        },
        getChildren: function (elt) {
            var children = elt.children;
            if (typeof children == 'undefined') {
                children = [];
                for (var i = 0; i < elt.childNodes.length; i++)
                    if (elt.childNodes[i].nodeName.indexOf('#') != 0)
                        children.push(elt.childNodes[i]);
            }
            return children;
        },
        handleGeocodeResponse: function (req) {
            var responseText = req.responseText;
            if (responseText && (responseText.indexOf('FAILED') != -1 ||
                responseText.indexOf('Exception') != -1)) {
                // fail silently
                return false;
            }
            //> Berend DWE/10042: req.responseXML is null on QA server, although we have valid xml markup in responseText
            if (responseText && !req.responseXML) {
                try {
                    req.responseXML = window.XMLDOM(responseText);
                } catch (e) {
                    return false;
                }
            }
            //< Berend DWE/10042
            //> Berend DWE/8257: support for nominatim.openstreetmap.org geocoder
            if (req.responseXML && req.responseXML.documentElement && (req.responseXML.documentElement.nodeName == 'searchresults')) {
                var results = this.getChildren(req.responseXML.documentElement); // Berend DWE/10042
                if (results && results.length > 0) {
                    // Reset search results if we got a new valid result
                    var oLayer = this.map.getLayersByName('locationtool')[0];
                    if (oLayer && (typeof oLayer != 'undefined'))
                        oLayer.removeAllFeatures();
                    this.resultFeatures = null;
                    // Use the bounding box of the first search result to zoom the map, if it has one.
                    var sBoundingBox = results[0].getAttribute('boundingbox');
                    if (sBoundingBox) {
                        var aBoundingBox = sBoundingBox.split(',');
                        if (aBoundingBox.length == 4) {
                            for (var i = 0; i < 4; i++) {
                                if (isNaN(aBoundingBox[i]))
                                    return false;
                                aBoundingBox[i] = parseFloat(aBoundingBox[i]);
                            }
                            // nominatim.openstreetmap.org returns bounding box as [latitudeBottom,latitudeTop,longitudeLeft,longitudeRight]
                            var bounds = new OpenLayers.Bounds(aBoundingBox[2], aBoundingBox[0], aBoundingBox[3], aBoundingBox[1]);
                            if (!this.proj4326) this.proj4326 = new OpenLayers.Projection("EPSG:4326");
                            bounds = bounds.transform(this.proj4326, this.map.getProjectionObject());
                            if (bounds)
                                this.map.zoomToExtent(bounds);
                        }

                        // If a single address was localized, the search result has type="house". Show a point for this result type.
                        var sType = results[0].getAttribute('type');
                        if (sType == 'house') {
                            var sLat = results[0].getAttribute('lat');
                            var sLon = results[0].getAttribute('lon');
                            if (!isNaN(sLat) && !isNaN(sLon)) {
                                var geom = new OpenLayers.Geometry.Point(parseFloat(sLon), parseFloat(sLat));
                                geom = geom.transform(this.proj4326, this.map.getProjectionObject());
                                var oResultFeature = { geometry: geom, attributes: { address: { street: '', building: '', postalCode: '', place: { Municipality: '' } } } };
                                //> Berend DWE/10042
                                var resultChildren = this.getChildren(results[0]);
                                for (var i = 0; i < resultChildren.length; i++) {
                                    var childNode = resultChildren[i];
                                    var sNodeText = childNode.textContent || childNode.innerText || childNode.childNodes[0].nodeValue; //< Berend DWE/10042
                                    switch (childNode.nodeName) {
                                        case 'house_number':
                                            oResultFeature.attributes.address.building = sNodeText;
                                            break;
                                        case 'road':
                                            oResultFeature.attributes.address.street = sNodeText;
                                            break;
                                        case 'postcode':
                                            oResultFeature.attributes.address.postalCode = sNodeText;
                                            break;
                                        case 'city':
                                        case 'town':
                                            oResultFeature.attributes.address.place.Municipality = sNodeText;
                                            break;
                                        case 'suburb':
                                            oResultFeature.attributes.address.place.MunicipalitySubdivision = sNodeText;
                                            break;
                                        case 'state':
                                            oResultFeature.attributes.address.place.CountrySubdivision = sNodeText;
                                            break;
                                        case 'country':
                                            oResultFeature.attributes.address.place.Country = sNodeText;
                                            break;
                                        case 'country_code':
                                            oResultFeature.attributes.address.place.CountryCode = sNodeText;
                                            break;
                                    }
                                }
                                var sResultText = (oResultFeature.attributes.address.street + ' ' + oResultFeature.attributes.address.building).trim()
                                    + ', ' + (oResultFeature.attributes.address.postalCode + ' ' + oResultFeature.attributes.address.place.Municipality).trim();
                                this.resultFeatures = [oResultFeature];
                                this.showResultPoint(0, sResultText);
                                this.hideResults();
                            }
                        }
                    }
                }
            } else {
                //< Berend DWE/8257
                var xlslusFormat = new Geozet.Format.XLSLUS();
                var xlslus = xlslusFormat.read(req.responseXML); // Berend issue 1435: req does not return responseText for performance reasons (Internet Explorer)
                if (xlslus.length == 0) {
                    hits = 0;
                } else {
                    var hits = xlslus[0].numberOfGeocodedAddresses;
                }
                if (hits == 0) {
                    // zero responses
                    if (!this.resultFeatures) {
                        //alert("Geen locaties gevonden ...");
                    }
                } else {
                    //> Berend DWE/8068: rewrite
                    this.decosSearchResultsId = null;
                    var maxEx = this.map.maxExtent; // Berend DWE/8257
                    var minx = maxEx.right;
                    var miny = maxEx.top;
                    var maxx = maxEx.left;
                    var maxy = maxEx.bottom;
                    var minzoom = 15;
                    var features = [];
                    // > 0 hit show suggestions
                    if (hits > 0) {
                        var searchString = this.searchString;
                        //var searchString = document.getElementById('pdoksearchinput').value;
                        document.getElementById('pdoksearchresults').innerHTML =
                            '<span class="closeWindow"><a onclick="return false;"><div class="closeWindow" alt="Sluiten" title="Sluiten"/>X</div></span></a>' +
                            '<span class="searchedFor">Gezocht op: "' + searchString + '"</span><h3>Zoekresultaten</h3><ul style="height:130px;overflow-y:auto" id="geozetSuggestions" class="geozetSuggestions"></ul>';
                    }
                    this.resultFeatures = xlslus[0].features;
                    var bInitResults = false;
                    for (var i = 0; i < hits; i++) {
                        var geom = this.resultFeatures[i].geometry;
                        var address = this.resultFeatures[i].attributes.address;
                        var plaats = address.place.MunicipalitySubdivision; // toont evt provincie afkorting
                        var gemeente = address.place.Municipality;
                        var prov = address.place.CountrySubdivision;
                        var adres = '';
                        var postcode = '';
                        // determine zoom and hash
                        var zoom = null;
                        if (address.street && address.street.length > 0) {
                            adres = address.street;
                            if (address.building) {
                                var toevoeging = '';
                                if (address.building.subdivision) {
                                    toevoeging = ' ' + address.building.subdivision;
                                    if (toevoeging.indexOf(' BS') == 0)
                                        toevoeging = (toevoeging.length == 3) ? ' bis' : toevoeging.replace(' BS', ' bis ');
                                }
                                adres += ' ' + address.building.number + toevoeging;
                                if ((i == 0) || !bInitResults) {
                                    bInitResults = true;
                                    if (parent && parent.mapSaveGeoSearchResults)
                                        this.decosSearchResultsId = parent.mapSaveGeoSearchResults(responseText);
                                    var oLayer = this.map.getLayersByName('locationtool')[0];
                                    if (oLayer && (typeof oLayer != 'undefined'))
                                        oLayer.removeAllFeatures();
                                }
                            }
                            if (!zoom) { zoom = 'adres' }
                        }
                        var plaatsNaam = '';
                        if (address.postalCode) {
                            plaatsNaam = address.postalCode + ' ';
                            if (!zoom) { zoom = 'postcode' }
                        }
                        if (plaats) {
                            plaatsNaam += plaats + ' (plaats)';
                            if (!zoom) { zoom = 'plaats' }
                        }
                        else if (gemeente) {
                            plaatsNaam += gemeente + ' (gemeente)';
                            if (!zoom) { zoom = 'gemeente' }
                        }
                        else if (prov) {
                            plaatsNaam += prov + ' (provincie)';
                            if (!zoom) { zoom = 'provincie' }
                        }
                        if (!zoom) { zoom = 'standaard' }
                        if (plaatsNaam) adres += ', ' + plaatsNaam;

                        // one hit? zoom to it. Otherwise show resultlist
                        if (hits > 1) {
                            // hack to be able to handle results without geom
                            var x = geom ? geom.x : 150000;
                            var y = geom ? geom.y : 450000;
                            var z = geom ? this.zoomScale[zoom] : this.zoomScale['provincie'];
                            var newId = -1;
                            if (geom) {
                                minx = Math.min(minx, x);
                                miny = Math.min(miny, y);
                                maxx = Math.max(maxx, x);
                                maxy = Math.max(maxy, y);
                                minzoom = Math.min(minzoom, this.zoomScale[zoom]);
                                //var newFt = new OpenLayers.Feature.Vector(new OpenLayers.Geometry.Point(x, y), { "title": suggestion, "postcode": postcode, "adres": adres, "plaats": plaats, "gemeente": gemeente, "provincie": prov });
                                ////> Berend DWE/8068: get an individual search result ID derived from the ID for the saved search results list
                                //if (parent && parent.mapGetGeoSearchResultId)
                                //  newFt.id = parent.mapGetGeoSearchResultId(this.decosSearchResultsId, i);
                                ////< Berend DWE/8068
                                newId = 'a_';
                                if (parent && parent.mapGetGeoSearchResultId)
                                    newId += parent.mapGetGeoSearchResultId(this.decosSearchResultsId, i);
                                else
                                    newId += i.toString();
                            }
                            var gazHtml = '<li><a href="#" style="color:blue;text-decoration:underline" id="' + newId + '" x="' + x + '" y="' + y + '" z="' + z + '" i="' + i + '"></a></li>'; // Berend DWE/8068
                            document.getElementById('geozetSuggestions').innerHTML += gazHtml;
                            var a = document.getElementById(newId);
                            if (a) {
                                if ('textContent' in a)
                                    a.textContent = adres;
                                else
                                    a.innerText = adres;
                            }
                            this.showResults();
                        } else {
                            this.showResultPoint(0, adres);
                            // hack to be able to handle results without geom
                            var x = geom ? geom.x : 150000;
                            var y = geom ? geom.y : 450000;
                            var z = geom ? this.zoomScale[zoom] : this.zoomScale['provincie'];
                            this.map.setCenter(new OpenLayers.LonLat(x, y), z);
                            this.hideResults();
                        }
                    }
                }
            }
            return false;
        },
        showResultPoint: function (i, sResultText) {
            if (this.resultFeatures && this.resultFeatures.length && (this.resultFeatures.length > i)) {
                var geom = this.resultFeatures[i].geometry;
                var address = this.resultFeatures[i].attributes.address;
                if (address.street && address.street.length > 0 && geom && sDecosAspxUrl && oDecosStyle2) {
                    var oLayer = this.map.getLayersByName('locationtool')[0];
                    if (oLayer && (typeof oLayer != 'undefined')) {
                        var newFt = new OpenLayers.Feature.Vector(new OpenLayers.Geometry.Point(geom.x, geom.y));
                        newFt.style = oDecosStyle2;
                        // get an individual search result ID derived from the ID for the saved search results list
                        if (parent && parent.mapGetGeoSearchResultId)
                            newFt.id = parent.mapGetGeoSearchResultId(this.decosSearchResultsId, i);
                        oLayer.addFeatures([newFt]);

                        // If the point button is avaiable in the parent window, enable it now so that
                        // only one click is needed to link the search result
                        if (parent && parent.mapSetPointButtonActive && parent.mapSetPointButtonActive(true))
                            DecosSelectPointActivate(true);
                    }
                }
            }
            // Description text of the search result is shown in the pdoksearchinput text box.
            if (sResultText) {
                var pdoksearchinput = document.getElementById('pdoksearchinput');
                if (pdoksearchinput) {
                    if (sResultText.lastIndexOf(')') == sResultText.length - 1) {
                        var iBracketPos = sResultText.lastIndexOf(' (');
                        if (iBracketPos > 0)
                            sResultText = sResultText.substr(0, iBracketPos);
                    }
                    pdoksearchinput.value = sResultText;
                }
            }
        },
        //< Berend DWE/8068
        search: function (searchString) {
            this.searchString = searchString;
            //> Berend 20130315
            if (searchString.length > 6) {
                var sPCHN = searchString.replace(/ /g, '');
                if (sPCHN.length > 6) {
                    var iPC = parseInt(searchString.substr(0, 4));
                    var sPC = searchString.substr(4, 2).toUpperCase();
                    var sHN = searchString.substr(6);
                    if (!isNaN(iPC) && (iPC >= 1000) && (iPC <= 9999) && (sPC >= 'AA') && (sPC <= 'ZZ')) // Berend DWE/8068
                        searchString = iPC.toString() + sPC + ' ' + sHN;
                }
            }
            //< Berend 20130415
            var params = {}; //{request: 'geocode'};
            params[this.geocoderParameter] = searchString;
            var proxy = (this.geocoderUrl.indexOf("https://") == 0) ? null : OpenLayers.ProxyHost; // Berend DWE/8257: https geocoder can be handled client-side
            if (searchString && searchString.length > 0) {
                OpenLayers.Request.GET({
                    url: this.geocoderUrl,
                    params: params,
                    proxy: proxy,
                    scope: this,
                    callback: this.handleGeocodeResponse
                    // failure: this.handleError
                });
            }
            return false;
        },

        hideResults: function () {
            document.getElementById('pdoksearchresults').style.display = 'none';
        },
        showResults: function () {
            //document.getElementById('pdoksearchresults').innerHTML = '';
            document.getElementById('pdoksearchresults').style.display = 'block';
        },

        CLASS_NAME: "OpenLayers.Control.GeocoderControl"
    });
