import { GlobalFunctions } from '../global-functions';
/**
 * Validate search criteria
 */
export class SearchHelper {
    /**
     * check if search query has valid Bsn number
     * @param searchQuery  search query
     * @returns boolean
     */
    static isValidBsnNumber(searchQuery) {
        const validBsnFormat = /^(\d{8}|\d{9})$/;
        if (searchQuery.match(validBsnFormat) == null) {
            return false;
        }
        const j = searchQuery.length;
        let pos = 0;
        let result = 0;
        for (let i = j; i > 0; i--) {
            result += (i !== 1) ? (searchQuery.charAt(pos) * i) : (searchQuery.charAt(pos) * i * -1);
            pos++;
        }
        return (result % 11 === 0);
    }
    /**
     * check for valid date number
     * @param searchQuery search query
     * @param datepipe date pipe
     * @returns  boolean
     */
    static isValidDateNumber(searchQuery, datepipe) {
        return (datepipe.transform(searchQuery.trim(), 'ddMMyy') != null
            || datepipe.transform(searchQuery.trim(), 'ddMMyyyy') != null
            || datepipe.transform(searchQuery.trim(), 'dd-MM-yy') != null
            || datepipe.transform(searchQuery.trim(), 'dd-MM-yyyy') != null
            || datepipe.transform(searchQuery.trim(), 'dd/MM/yy') != null
            || datepipe.transform(searchQuery.trim(), 'dd/MM/yyyy') != null);
    }
    /**
     * check for valid search text
     * @param searchQuery search query
     * @returns boolean
     */
    static isValidText(searchQuery) {
        const validTextFormat = /^\s*([a-zA-Z ]+)\s*$/;
        return searchQuery.match(validTextFormat) != null;
    }
    /**
     * check for valid number
     * @param searchQuery search query
     * @returns boolean
     */
    static isValidNumber(searchQuery) {
        const validNumberformat = /^\d+$/;
        return searchQuery.match(validNumberformat) != null && !this.isValidBsnNumber(searchQuery);
    }
    /**
     * check for valid case is
     * @param searchQuery search query
     * @returns boolean
     */
    static isValidCaseId(searchQuery) {
        const validcaseIdFormat = /^(z|z-|z\/)[0-9]|^\d{4}\//i;
        return searchQuery.match(validcaseIdFormat) != null;
    }
    /**
     * check for valid kvk number
     * @param searchQuery search query
     * @returns boolean
     */
    static isValidKvkNumber(searchQuery) {
        const validKvkFormat = /^[Kk](\d{8}|\d{9})$/;
        return searchQuery.match(validKvkFormat) != null;
    }
    /**
     * check for valid person search criteria
     * @param searchQuery search query
     * @returns boolean
     */
    static isValidPersonSearchCriteria(searchQuery, datepipe) {
        return (this.isValidBsnNumber(searchQuery)
            || this.isValidDateNumber(searchQuery, datepipe)
            || this.isValidZipcodeAndHousenumber(searchQuery)
            || this.isValidCombinationSearch(searchQuery, datepipe)
            || GlobalFunctions.isValidEmailAddress(searchQuery)
            || this.isValidContactnumber(searchQuery));
    }
    /**
     * check for valid product search criteria
     * @param searchQuery search query
     * @returns boolean
     */
    static isValidProductSearchCriteria(searchQuery) {
        return this.isValidText(searchQuery);
    }
    /**
     * check for valid question answer search criteria
     * @param searchQuery search query
     * @returns boolean
     */
    static isValidQuestionAnswerSearchCriteria(searchQuery) {
        return this.isValidText(searchQuery);
    }
    /**
     * check for valid organization search criteria
     * @param searchQuery search query
     * @returns boolean
     */
    static isValidOrganizationSearchCriteria(searchQuery) {
        return this.isValidText(searchQuery) || this.isValidKvkNumber(searchQuery);
    }
    /**
     * check for valid case search criteria
     * @param searchQuery search query
     * @returns boolean
     */
    static isValidCaseSearchCriteria(searchQuery) {
        return this.isValidCaseId(searchQuery);
    }
    /**
     * check for valid source search criteria
     * @param searchQuery search query
     * @param datepipe date pipe
     * @returns boolean
     */
    static isValidAllSourceSearchCriteria(searchQuery, datepipe) {
        return (this.isValidBsnNumber(searchQuery)
            || this.isValidDateNumber(searchQuery, datepipe)
            || this.isValidText(searchQuery)
            || this.isValidCaseId(searchQuery));
    }
    /**
     * check for valid zipcode houseno search criteria
     * @param searchQuery search query
     * @returns boolean
     */
    static isValidZipcodeAndHousenumber(searchQuery) {
        const zipcode = /^([0-9]{4}[a-z|A-Z]{2})|([0-9]{4}[ ][a-z|A-Z]{2})\b/;
        return searchQuery.match(zipcode) != null;
    }
    /**
     *
     * @param searchQuery search query
     * @param datepipe datepipe
     * @returns
     * [check]- validate if elemnet.value have some value
     */
    static isValidCombinationSearch(searchQuery, datepipe) {
        const validMixedFormat = /^[A-Za-z0-9 ]+$/;
        let containsDate;
        let containSurname;
        let containEmailOrPhoneOrMobile;
        if (searchQuery.indexOf(' ') !== -1) {
            const dateparts = searchQuery.split(' ', 2);
            dateparts.forEach((element) => {
                if (this.isValidDateNumber(element.value, datepipe)) {
                    containsDate = true;
                }
                else if (GlobalFunctions.isValidEmailAddress(element.value) || this.isValidContactnumber(element.value)) {
                    containEmailOrPhoneOrMobile = true;
                }
                else {
                    containSurname = true;
                }
            });
        }
        return (searchQuery.match(validMixedFormat) != null && containsDate) || (containSurname && containEmailOrPhoneOrMobile);
    }
    /**
     * check for valid person search criteria with text
     * @param searchQuery search query
     * @param datepipe date pipe
     * @returns boolean
     */
    static isValidPersonSearchCriteriaWithText(searchQuery, datepipe) {
        return this.isValidPersonSearchCriteria(searchQuery, datepipe) || this.isValidText(searchQuery);
    }
    /**
     * check for valid case search criteria with text
     * @param searchQuery search query
     * @returns boolean
     */
    static isValidCaseSearchCriteriaWithText(searchQuery) {
        return this.isValidCaseId(searchQuery) || this.isValidText(searchQuery);
    }
    /**
     * check for valid document search criteria with text
     * @param searchQuery search query
     * @returns boolean
     */
    static isValidDecoumentSearchCriteriaWithText(searchQuery) {
        return this.isValidNumber(searchQuery) || this.isValidText(searchQuery);
    }
    /**
     * check for valid contanct number
     * @param searchQuery search query
     * @returns boolean
     */
    static isValidContactnumber(searchQuery) {
        searchQuery = searchQuery.replace(/" "/g, '');
        const validContactnumberFormat = /(^\+31|^31|^00|^0)([0-9]{9})$/;
        return searchQuery.match(validContactnumberFormat) != null;
    }
}
//# sourceMappingURL=search-helper.js.map