import { formatDate } from '@angular/common';
import dayGridPlugin from '@fullcalendar/daygrid';
import interactionPlugin from '@fullcalendar/interaction';
import listPlugin from '@fullcalendar/list';
import momentPlugin from '@fullcalendar/moment';
import timeGridPlugin from '@fullcalendar/timegrid';
import { parse } from 'date-fns';
import { cloneDeep } from "lodash";
import moment from 'moment';
import { AgendaShowAsStatus, ChatState, FreeBusyStatus, LicenseSpecificFeature, OperationType, PrivilegeScope, ResolutionType, StakeholderType } from '../core/enums/common-enums';
import { EntityFqdn } from '../core/enums/core-enums';
import { EmailSource } from '../core/enums/global-enum';
import { CallStatus } from '../core/enums/phone-enum';
import { TaskHandlerType } from '../core/enums/task-enum';
import { ActionMenu } from '../core/models/common/action-menu';
import { BaseDto } from '../core/models/common/baseDto';
import { RequiredFields } from '../core/models/common/required-fields';
import { SearchObject } from '../core/models/common/search-objectDto';
import { ContentType, JsonDtoType, TaskDtoSource } from '../core/models/jkc-object/jkc-object';
import { ResolutionInfo } from '../core/models/resolution/resolution-info';
import { TaskCustomerInformationDto } from '../core/models/task/task-customer-informationDto';
import { DateTimeFormat, Domain, LocalStorageKey } from './constants';
import { CommonHelper } from './helper/common-helper';
let X2JS = require('x2js');
export class GlobalFunctions {
    static getGuid() {
        return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, (c) => {
            const r = Math.random() * 16 | 0, v = c === 'x' ? r : (r & 0x3 | 0x8);
            return v.toString(16);
        });
    }
    static setDateFormat(elementName, date, dateformatString, datePipe) {
        try {
            date = new Date(date);
            if (elementName != null && GlobalFunctions.validDate(date)) {
                const inputElement = document.getElementById(elementName);
                if (inputElement) {
                    inputElement.value = datePipe.transform(date, dateformatString, "", CommonHelper.getLanguageCode());
                    inputElement.value = this.removeDotAfterMonth(inputElement.value);
                    return inputElement.value;
                }
            }
        }
        catch (e) {
            console.error(e.message);
        }
    }
    static getFormattedDateString(date, format, locale) {
        return formatDate(date, format, locale);
    }
    static isEmpty(str) {
        return (str === undefined || str == null || str.length <= 0) ? true : false;
    }
    static isValidEmailAddress(emailAddress) {
        if (!this.isEmpty(emailAddress))
            return /^\w+@[a-zA-Z_]+?\.[a-zA-Z]{2,3}$/.test(emailAddress);
    }
    static setCookie(key, value) {
        localStorage.setItem(key, value);
    }
    static getCookie(key) {
        return localStorage.getItem(key);
    }
    static removeCookie(key) {
        localStorage.removeItem(key);
    }
    static showCurrentHandlerTitle(value) {
        return value.split(" - ").join("\n").trim();
    }
    /**
     * encrypt or decrypt password
     * @param inputString  password in raw string
     * @returns  encrypted/decrypted password
     */
    static encryptOrDecryptUsingXOR(inputString) {
        let result = "";
        if (!this.isEmpty(inputString) && inputString.length > 0) {
            for (let i = 0; i < inputString.length; i++) {
                // create block
                let a = inputString.charCodeAt(i);
                // bitwise XOR
                let b = a ^ 9;
                result = result + String.fromCharCode(b);
            }
        }
        return result;
    }
    ;
    static hasRight(fullyQualifiedName, operation, objectSource = null) {
        let hasRight = null;
        let privilegeItems = null;
        let objectSourcePrivilegeItems = [];
        let privilegeData = this.userPrivilegeData;
        if (privilegeData != undefined && privilegeData != null) {
            if (fullyQualifiedName == null || fullyQualifiedName == '') {
                privilegeItems = [];
                if (privilegeData.GeneralPrivileges != null) {
                    for (let i = 0; i < privilegeData.GeneralPrivileges.length; i++) {
                        if (privilegeData.GeneralPrivileges[i].Privilege.OperationType == operation) {
                            privilegeItems.push(privilegeData.GeneralPrivileges[i]);
                        }
                    }
                }
            }
            else {
                let entityPrivilege = null;
                if (privilegeData.EntityPrivileges != null) {
                    for (let i = 0; i < privilegeData.EntityPrivileges.length; i++) {
                        if ((privilegeData.EntityPrivileges[i].ObjectSource == null || privilegeData.EntityPrivileges[i].ObjectSource == '') && privilegeData.EntityPrivileges[i].EntityFqdn != undefined && fullyQualifiedName != undefined && privilegeData.EntityPrivileges[i].EntityFqdn.toUpperCase() == fullyQualifiedName.toUpperCase()) {
                            entityPrivilege = privilegeData.EntityPrivileges[i];
                            break;
                        }
                    }
                    if (entityPrivilege != null) {
                        privilegeItems = [];
                        if (entityPrivilege.privilegeItems != null) {
                            for (let i = 0; i < entityPrivilege.privilegeItems.length; i++) {
                                if (entityPrivilege.privilegeItems[i].Privilege.OperationType == operation) {
                                    if (entityPrivilege.privilegeItems[i].ObjectSource == null)
                                        privilegeItems.push(entityPrivilege.privilegeItems[i]);
                                    else
                                        objectSourcePrivilegeItems.push(entityPrivilege.privilegeItems[i]);
                                }
                            }
                        }
                    }
                }
            }
            if (privilegeItems != null) {
                for (let i = 0; i < privilegeItems.length; i++) {
                    if (hasRight == null || (privilegeItems[i].PrivilegeScope == PrivilegeScope.UserSpecific && privilegeItems[i].Allow == false) || (privilegeItems[i].PrivilegeScope == PrivilegeScope.UserGroupSpecific && hasRight == true))
                        hasRight = privilegeItems[i].Allow;
                }
            }
            if (objectSourcePrivilegeItems.length > 0) {
                if (!this.isEmpty(objectSource)) {
                    for (let index = 0; index < objectSourcePrivilegeItems.length; index++) {
                        if (objectSourcePrivilegeItems[index].ObjectSource.toUpperCase() == objectSource.toUpperCase()) {
                            hasRight = objectSourcePrivilegeItems[index].Allow;
                            break;
                            ;
                        }
                    }
                }
                else if (hasRight != true) {
                    for (let index = 0; index < objectSourcePrivilegeItems.length; index++) {
                        if (objectSourcePrivilegeItems[index].Allow) {
                            hasRight = true;
                            break;
                            ;
                        }
                    }
                }
            }
        }
        return hasRight;
    }
    ;
    static getSessionTokenAsString() {
        return JSON.parse(this.getCookie(LocalStorageKey.SESSION_TOKEN))?.Id ?? null;
    }
    ;
    static isLicenseSpecificFeatureEnabled(entity) {
        let isEnabled = true;
        if (this.licenseSpecificFeatureList != null && this.licenseSpecificFeatureList.length > 0) {
            for (let i = 0; i < this.licenseSpecificFeatureList.length; i++) {
                if (this.licenseSpecificFeatureList[i].LicenseTypeFeature == entity) {
                    isEnabled = this.licenseSpecificFeatureList[i].LicenseSpecificFeatureResult == 0 ? true : false;
                    break;
                }
            }
        }
        return isEnabled;
    }
    ;
    static getLoggedInUserId() {
        return JSON.parse(this.getCookie(LocalStorageKey.SESSION_TOKEN))?.UserId ?? '';
    }
    static toContentType(contentType) {
        let oldContentType = contentType;
        if (!this.isEmpty(contentType)) {
            switch (contentType.toUpperCase()) {
                case "CASEENTITY":
                    {
                        oldContentType = ContentType.Case;
                        break;
                    }
                case "CASETYPE":
                case "CASETYPES":
                    {
                        oldContentType = ContentType.CaseType;
                        break;
                    }
                case "DOCUMENT":
                case "DOCUMENTS":
                    {
                        oldContentType = ContentType.Document;
                        break;
                    }
                case "EMAIL":
                case "EMAILS":
                    {
                        oldContentType = ContentType.Email;
                        break;
                    }
                case "EMPLOYEE":
                case "EMPLOYEES":
                    {
                        oldContentType = ContentType.Employee;
                        break;
                    }
                case "NOTEENTITY":
                    {
                        oldContentType = ContentType.Note;
                        break;
                    }
                case "ORGANIZATION":
                case "ORGANIZATIONS":
                    {
                        oldContentType = ContentType.Organization;
                        break;
                    }
                case "PERSON":
                case "PERSONS":
                    {
                        oldContentType = ContentType.Person;
                        break;
                    }
                case "PRODUCT":
                case "PRODUCTS":
                    {
                        oldContentType = ContentType.Product;
                        break;
                    }
                case "TASK":
                case "TASKS":
                    {
                        oldContentType = ContentType.Task;
                        break;
                    }
                case "CALLBACKNOTE":
                    {
                        oldContentType = ContentType.CallbackNote;
                        break;
                    }
                case "APPOINTMENT":
                    {
                        oldContentType = ContentType.Appointment;
                        break;
                    }
                case "ASSIGNMENT":
                    {
                        oldContentType = ContentType.Assignment;
                        break;
                    }
                case "REPORTIMPROVEMENT":
                    {
                        oldContentType = ContentType.ReportImprovement;
                        break;
                    }
                case "INFORMATIONREQUEST":
                    {
                        oldContentType = ContentType.InformationRequest;
                        break;
                    }
                case "INFORMATIONRESPONSE":
                    {
                        oldContentType = ContentType.InformationResponse;
                        break;
                    }
                case "DEPARTMENT":
                case "DEPARTMENTS":
                    {
                        oldContentType = ContentType.Department;
                        break;
                    }
                case "CONTACTMOMENT":
                case "CONTACTHISTORY":
                    {
                        oldContentType = ContentType.ContactMoment;
                        break;
                    }
                case "ITEM":
                case "NEWS":
                    {
                        oldContentType = ContentType.News;
                        break;
                    }
                case "QUESTIONANSWER":
                case "QUESTIONANSWERS":
                    {
                        oldContentType = ContentType.QuestionAnswer;
                        break;
                    }
                case "SESSION":
                    {
                        oldContentType = ContentType.Session;
                        break;
                    }
                case "LOCATION":
                case "LOCATIONS":
                    {
                        oldContentType = ContentType.Location;
                        break;
                    }
                case "FILEENTITY":
                    {
                        oldContentType = ContentType.File;
                        break;
                    }
                case "THEMEPAGE":
                case "THEMEPAGES":
                case "THEME":
                case "THEMES":
                    {
                        oldContentType = ContentType.Theme;
                        break;
                    }
                case "LINK":
                case "LINKS":
                    {
                        oldContentType = ContentType.Link;
                        break;
                    }
                case "SESSIONNOTE":
                    {
                        oldContentType = ContentType.SessionNote;
                        break;
                    }
                case "FACEBOOK":
                case "DECOS.KCS.SOCIALMEDIA.ENTITIES.FACEBOOK":
                    {
                        oldContentType = ContentType.Facebook;
                        break;
                    }
                case "DECOS.KCS.SOCIALMEDIA.ENTITIES.TWITTER":
                case "TWITTER":
                    {
                        oldContentType = ContentType.Twitter;
                        break;
                    }
                case "DECOS.KCS.SOCIALMEDIA.ENTITIES.WHATSAPP":
                case "WHATSAPP":
                    {
                        oldContentType = ContentType.WhatsApp;
                        break;
                    }
                case "SOCIALMEDIA":
                    {
                        oldContentType = ContentType.SocialMedia;
                        break;
                    }
                case "TICKET":
                case "TICKETS":
                    {
                        oldContentType = ContentType.Ticket;
                        break;
                    }
                case "ISSUE":
                case "ISSUES":
                    {
                        oldContentType = ContentType.Issue;
                        break;
                    }
                case "CHAT":
                    {
                        oldContentType = ContentType.Chat;
                        break;
                    }
                case "ALLSOURCES":
                    {
                        oldContentType = ContentType.AllSources;
                        break;
                    }
                case "DOMAIN":
                case "DOMAINS":
                    {
                        oldContentType = ContentType.Domain;
                        break;
                    }
            }
        }
        return oldContentType || '';
    }
    ;
    static setPrivilegeData(userPrivilegeData, sessionManager, translateService) {
        this.userPrivilegeData = userPrivilegeData;
        sessionManager.actionMenu = new ActionMenu(translateService);
        sessionManager.showTopHeader = this.checkRights("", OperationType.ShowTopHeader);
        sessionManager.showManagementConsole = this.checkRights("", OperationType.ManagementConsoleAccess);
        sessionManager.showTaskReminder = this.checkRights(GlobalFunctions.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Tasks_Entities_Task]), OperationType.SendTaskReminder);
        sessionManager.hasViewerAccess = this.checkRights("", OperationType.Viewer);
    }
    static checkRights(contentType, operation = OperationType.View) {
        if (!this.isEmpty(contentType)) {
            if (contentType.toUpperCase() == "CALLBACKNOTE" || contentType.toUpperCase() == "APPOINTMENT" || contentType.toUpperCase() == "ASSIGNMENT" || contentType.toUpperCase() == "REPORTIMPROVEMENT" || contentType.toUpperCase() == "INFORMATIONREQUEST" || contentType.toUpperCase() == "INFORMATIONRESPONSE")
                contentType = "TASK";
            if (contentType.toUpperCase() == "WHATSAPP" || contentType.toUpperCase() == "TWITTER" || contentType.toUpperCase() == "FACEBOOK")
                contentType = "SOCIALMEDIA";
        }
        let oldContentType = this.toContentType(contentType);
        return this.hasRight(oldContentType, operation);
    }
    static getAllowedDataRights(fullyQualifiedName, operationType) {
        let allowedData = "";
        if (!this.isEmpty(fullyQualifiedName) && !this.isEmpty(this.userPrivilegeData) && !this.isEmpty(this.userPrivilegeData.EntityPrivileges)) {
            for (let i = 0; i < this.userPrivilegeData.EntityPrivileges.length; i++) {
                if (this.userPrivilegeData.EntityPrivileges[i].EntityFqdn.toUpperCase() == fullyQualifiedName.toUpperCase()) {
                    switch (fullyQualifiedName.toUpperCase()) {
                        case ContentType.Person.toUpperCase():
                        case ContentType.Organization.toUpperCase():
                        case ContentType.Employee.toUpperCase():
                            {
                                let privilegeItems = this.userPrivilegeData.EntityPrivileges[i].privilegeItems;
                                for (let j = 0; j < privilegeItems.length; j++) {
                                    if (privilegeItems[j].Privilege.OperationType == operationType) {
                                        allowedData = privilegeItems[j].AllowedData;
                                        break;
                                    }
                                }
                                break;
                            }
                    }
                    break;
                }
            }
        }
        return allowedData;
    }
    static setGrammarForBaseDto(obj, grammar) {
        if (grammar == null || obj == null)
            return;
        let startReading = false;
        let keepReading = true;
        let grammarValue = "";
        let lastActualGrammarValue = "";
        let temp = "";
        for (let i = 0; i < grammar.length; i++) {
            if (grammar[i] == '{') {
                startReading = true;
                keepReading = false;
                temp += grammar[i];
            }
            else if (grammar[i] == '}') {
                temp += grammar[i];
                startReading = false;
                keepReading = true;
                let value = this.getPlaceholderValue(obj, temp);
                if (this.isEmpty(value)) {
                    value = "";
                    if (i == grammar.Length - 1)
                        grammarValue = lastActualGrammarValue;
                }
                grammarValue += value;
                lastActualGrammarValue = grammarValue;
                temp = "";
            }
            else if (startReading)
                temp += grammar[i];
            if (keepReading && grammar[i] != '}')
                grammarValue += grammar[i];
        }
        grammarValue = grammarValue.replace(/(^[,\s]+)|([,\s]+$)|[\s*][,\s][\s*]/g, '');
        return grammarValue;
    }
    static setValueByPropertyName(propertyName, source, value) {
        if (!(this.isEmpty(propertyName) && (this.isEmpty(source)) && (this.isEmpty(value)))) {
            source[propertyName] = value;
        }
    }
    static hasAllowDataRights(fullyQualifiedName, operationType, id) {
        let hasRight = false;
        if (this.isEmpty(fullyQualifiedName) || this.isEmpty(this.userPrivilegeData) || this.isEmpty(this.userPrivilegeData.EntityPrivileges) || this.isEmpty(id))
            return hasRight;
        for (let i = 0; i < this.userPrivilegeData.EntityPrivileges.length; i++) {
            if (this.userPrivilegeData.EntityPrivileges[i].EntityFqdn.toUpperCase() == fullyQualifiedName.toUpperCase()) {
                switch (fullyQualifiedName.toUpperCase()) {
                    case ContentType.Person.toUpperCase():
                    case ContentType.Organization.toUpperCase():
                    case ContentType.Email.toUpperCase():
                        {
                            let privilegeItems = this.userPrivilegeData.EntityPrivileges[i].privilegeItems;
                            for (let j = 0; j < privilegeItems.length; j++) {
                                if (privilegeItems[j].Privilege.OperationType == operationType) {
                                    let dataIds = privilegeItems[j].AllowedData;
                                    if (privilegeItems[j].Allow && !this.isEmpty(dataIds))
                                        hasRight = dataIds.indexOf(id) >= 0;
                                    break;
                                }
                            }
                            break;
                        }
                }
                break;
            }
        }
        return hasRight;
    }
    static getPlaceholderValue(obj, propertyName) {
        let result = obj;
        if (!this.isEmpty(propertyName)) {
            propertyName = propertyName.substring(1, propertyName.length - 1);
            let propertyList = propertyName.split('.');
            try {
                propertyList.forEach(tempValue => {
                    result = this.getPropertyValue(result, tempValue);
                    //TODO code for parsing the datetime yet to be written
                });
            }
            catch (e) {
                result = obj;
            }
        }
        return result;
    }
    static getPropertyValue(obj, property) {
        let value;
        let propertyAndIndex = property.split('[');
        if (propertyAndIndex.length > 1) {
            let index = propertyAndIndex[1].split(']')[0];
            value = obj[propertyAndIndex[0]][index];
        }
        else
            value = obj[propertyAndIndex];
        return value;
    }
    ;
    static getObjectTypeName(relatedEntityFqdnValue) {
        let result = '';
        if (!this.isEmpty(relatedEntityFqdnValue)) {
            if (!isNaN(relatedEntityFqdnValue))
                relatedEntityFqdnValue = EntityFqdn[relatedEntityFqdnValue];
            let entityLastName = relatedEntityFqdnValue.split('_');
            if (!this.isEmpty(entityLastName) && entityLastName.length > 1)
                result = entityLastName[entityLastName.length - 1];
            else {
                entityLastName = relatedEntityFqdnValue.split('.');
                if (!this.isEmpty(entityLastName) && entityLastName.length > 0)
                    result = entityLastName[entityLastName.length - 1];
            }
        }
        return result;
    }
    static parseXmlValue(xmlValue, key) {
        try {
            const xmlDoc = $.parseXML(xmlValue), $xml = $(xmlDoc), xmlValueResult = $xml.find('[Key=' + key + ']').attr('Value');
            return xmlValueResult;
        }
        catch (e) {
            console.error(e);
            return '';
        }
    }
    static getThemeSourceSetting(inputString) {
        let themeSources = [];
        let x2js = new X2JS();
        let themeSourceConfig = x2js.xml2js(inputString);
        if (themeSourceConfig != null && themeSourceConfig.ThemeSources != null && themeSourceConfig.ThemeSources.ThemeSource != null) {
            if (Array.isArray(themeSourceConfig.ThemeSources.ThemeSource))
                themeSourceConfig.ThemeSources.ThemeSource.forEach(themeSource => themeSources.push(themeSource));
            else if (typeof themeSourceConfig.ThemeSources.ThemeSource == 'object')
                themeSources.push(themeSourceConfig.ThemeSources.ThemeSource);
        }
        return themeSources;
    }
    static ConvertToArrayIfOnlySingleObject(arrayObject) {
        if (arrayObject.length == null) {
            let tempObj = [];
            tempObj = cloneDeep(arrayObject);
            arrayObject = [];
            arrayObject.push(tempObj);
        }
        return arrayObject;
    }
    ;
    static getCallerName(personDetails, grammervalue) {
        if (personDetails) {
            if (!GlobalFunctions.isEmpty(grammervalue))
                return this.setGrammarForBaseDto(personDetails, grammervalue);
            else
                return (this.isEmpty(personDetails.Details.AddressName) ? "" : (this.isEmpty(personDetails.Details.AddressName.Initials) ? "" : personDetails.Details.AddressName.Initials + " "))
                    + "" + (this.isEmpty(personDetails.Details.FormalName.Prefix) ? "" : personDetails.Details.FormalName.Prefix + " ")
                    + "" + personDetails.Details.FormalName.Surname;
        }
    }
    static asc2hex(sourceString) {
        let destinationstr = '';
        for (let a = 0; a < sourceString.length; a = a + 1) {
            destinationstr = destinationstr + sourceString.charCodeAt(a).toString(16);
        }
        return destinationstr;
    }
    static setStringPresentation(item, grammerValue, datePipe, defaultDateFormat) {
        if (!this.isEmpty(grammerValue) && !this.isEmpty(item)) {
            let value = this.setGrammarForBaseDto(item, grammerValue);
            this.setValueByPropertyName("StringRepresentation", item, value);
        }
    }
    static hasDateInStringRepresentation(grammarValue, datePipe, defaultDateFormat) {
        let dateFound = false;
        let newVal = "";
        let breadcrumbTitleList = grammarValue.split(",");
        const datePattern = /\b(\d{1,2}[-/]\d{1,2}[-/]\d{4}|\d{4}[-/]\d{1,2}[-/]\d{1,2}|\d{1,2} \w{3} \d{4}|\d{1,2}\/\d{1,2}\/\d{4})\b/g;
        breadcrumbTitleList.forEach((title, index) => {
            const matches = title.match(datePattern);
            if (matches) {
                matches.forEach((dateString) => {
                    let dateObject;
                    dateObject = this.tryParseToDateString(dateString, datePipe, defaultDateFormat);
                    if (!(dateObject.toString() === 'Invalid Date')) {
                        dateFound = true;
                        let formattedDate;
                        if (datePipe != undefined)
                            formattedDate = datePipe.transform(dateObject, defaultDateFormat) ?? dateString;
                        else {
                            const dt = new Date(dateObject);
                            formattedDate = dt.toLocaleDateString(CommonHelper.getLanguageCode(), { day: "numeric", month: "long", year: "numeric", hour: "2-digit", minute: "2-digit" });
                        }
                        title = title.replace(dateString, formattedDate);
                    }
                });
            }
            newVal += (index === breadcrumbTitleList.length - 1) ? title : title + ",";
        });
        return dateFound ? newVal : grammarValue;
    }
    static getLastPersonFromSession(list) {
        let result = null;
        if (list != null && list != "undefined" && list.length > 0) {
            for (let i = list.length - 1; i >= 0; i--) {
                if (list[i] != null && list[i].value != null && list[i].value.indexOf("Person") != -1) {
                    result = list[i].key;
                    break;
                }
            }
        }
        return result;
    }
    static isPrivateField(field, contentType) {
        const privateFields = this.getAllowedDataRights(contentType, OperationType.PrivateField);
        const isPrivate = (!this.isEmpty(privateFields) && privateFields.indexOf("|" + field + "|") !== -1);
        return isPrivate;
    }
    static containsObject(obj, list) {
        let i;
        for (i = 0; i < list.length; i++) {
            if (list[i] != null && list[i].key != null && obj != null) {
                const sourceId = this.getValueByPropertyName("Id", obj);
                const targetId = this.getValueByPropertyName("Id", list[i].key);
                if (sourceId == targetId)
                    return true;
            }
        }
        return false;
    }
    static setEntityType(entity, type) {
        let jsonString = JSON.stringify(entity);
        let jsonType = type;
        let startsWithArray = false;
        if (jsonString.indexOf("[") == 0)
            startsWithArray = true;
        if (!startsWithArray)
            jsonString = jsonString.substr(1);
        else
            jsonString = jsonString.substr(2);
        jsonString = jsonType + jsonString;
        if (startsWithArray)
            jsonString = jsonString.substring(0, jsonString.length - 1);
        entity = JSON.parse(jsonString);
        return entity;
    }
    static toJsonFqdnType(fqdnType) {
        let jsonFqdnTypeText = "";
        switch (fqdnType.toUpperCase()) {
            case "CASEENTITY":
                {
                    jsonFqdnTypeText = JsonDtoType.Case;
                    break;
                }
            case "CASETYPE":
                {
                    jsonFqdnTypeText = JsonDtoType.CaseType;
                    break;
                }
            case "DOCUMENT":
                {
                    jsonFqdnTypeText = JsonDtoType.Document;
                    break;
                }
            case "EMAIL":
                {
                    jsonFqdnTypeText = JsonDtoType.Email;
                    break;
                }
            case "EMPLOYEE":
                {
                    jsonFqdnTypeText = JsonDtoType.Employee;
                    break;
                }
            case "NOTEENTITY":
                {
                    jsonFqdnTypeText = JsonDtoType.Note;
                    break;
                }
            case "ORGANIZATION":
                {
                    jsonFqdnTypeText = JsonDtoType.Organization;
                    break;
                }
            case "PERSON":
                {
                    jsonFqdnTypeText = JsonDtoType.Person;
                    break;
                }
            case "PRODUCT":
                {
                    jsonFqdnTypeText = JsonDtoType.Product;
                    break;
                }
            case "QUESTIONANSWER":
                {
                    jsonFqdnTypeText = JsonDtoType.QuestionAnswer;
                    break;
                }
            case "CALLBACKNOTE":
                {
                    jsonFqdnTypeText = JsonDtoType.CallbackNote;
                    break;
                }
            case "ASSIGNMENT":
                {
                    jsonFqdnTypeText = JsonDtoType.Assignment;
                    break;
                }
            case "TASK":
                {
                    jsonFqdnTypeText = JsonDtoType.Task;
                    break;
                }
            case "EMAILATTACHMENT":
                {
                    jsonFqdnTypeText = JsonDtoType.EmailAttachment;
                    break;
                }
            case "FILEDOCUMENT":
                {
                    jsonFqdnTypeText = JsonDtoType.FileDocument;
                    break;
                }
            case "DEPARTMENT":
                {
                    jsonFqdnTypeText = JsonDtoType.Department;
                    break;
                }
            case "CONTACTMOMENT":
                {
                    jsonFqdnTypeText = JsonDtoType.ContactMoment;
                    break;
                }
            case "THEME":
                {
                    jsonFqdnTypeText = JsonDtoType.Theme;
                    break;
                }
            case "SOCIALMEDIA":
                {
                    jsonFqdnTypeText = JsonDtoType.SocialMedia;
                    break;
                }
            case "WHATSAPP":
                {
                    jsonFqdnTypeText = JsonDtoType.WhatsApp;
                    break;
                }
            case "TWITTER":
                {
                    jsonFqdnTypeText = JsonDtoType.Twitter;
                    break;
                }
            case "FACEBOOK":
                {
                    jsonFqdnTypeText = JsonDtoType.Facebook;
                    break;
                }
            case "DOMAIN":
                {
                    jsonFqdnTypeText = JsonDtoType.Domain;
                    break;
                }
        }
        return jsonFqdnTypeText;
    }
    static convertTaskContentType(contentType) {
        let convertedContentType = contentType;
        if (!GlobalFunctions.isEmpty(contentType) && EntityFqdn[contentType] !== undefined) {
            switch (EntityFqdn[contentType].toString()) {
                case EntityFqdn.Decos_Kcs_Tasks_Entities_CallbackNote.toString():
                case EntityFqdn.Decos_Kcs_Tasks_Entities_Appointment.toString():
                case EntityFqdn.Decos_Kcs_Tasks_Entities_Assignment.toString():
                case EntityFqdn.Decos_Kcs_Tasks_Entities_ReportImprovement.toString():
                case EntityFqdn.Decos_Kcs_Tasks_Entities_InformationRequest.toString():
                    {
                        convertedContentType = EntityFqdn[EntityFqdn.Decos_Kcs_Tasks_Entities_Task];
                        break;
                    }
            }
        }
        return convertedContentType;
    }
    ;
    static convertStakeHolderTypeToContentType(stakeholderType) {
        let contentType = "";
        switch (stakeholderType) {
            case StakeholderType.Employee:
                {
                    contentType = this.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Employees_Entities_Employee]);
                    break;
                }
            case StakeholderType.Person:
                {
                    contentType = this.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Persons_Entities_Person]);
                    break;
                }
            case StakeholderType.Organisation:
                {
                    contentType = this.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Organizations_Entities_Organization]);
                    break;
                }
        }
        return contentType;
    }
    static getRelatedObjects(loadedData, from = null) {
        let contentType = this.convertTaskContentType(loadedData.RelatedEntityFqdn);
        let relatedEntityFqdn = EntityFqdn[contentType];
        contentType = this.toString(contentType);
        let mainObject = null;
        let nativeObjectList = [];
        let relatedObjectInformation = {};
        if (!this.isEmpty(relatedEntityFqdn)) {
            switch (relatedEntityFqdn.toString()) {
                case EntityFqdn.Decos_Kcs_Persons_Entities_Person.toString():
                    {
                        if (from != null && (from == EntityFqdn.Decos_Kcs_Issues_Entities_Issue || from == EntityFqdn.Decos_Kcs_Email_Entities_Email))
                            mainObject = new SearchObject(loadedData.Id, contentType, loadedData.ObjectSource, loadedData.EmailAddress);
                        else {
                            if (!this.isEmpty(loadedData.AdministeredPersonDto))
                                mainObject = new SearchObject(loadedData.Id, contentType, loadedData.ObjectSource, loadedData.AdministeredPersonDto.ServiceNumber);
                            else
                                mainObject = new SearchObject(loadedData.Id, contentType, loadedData.ObjectSource, "");
                        }
                        break;
                    }
                case EntityFqdn.Decos_Kcs_Organizations_Entities_Organization.toString():
                    {
                        if (from != null && (from == EntityFqdn.Decos_Kcs_Issues_Entities_Issue || from == EntityFqdn.Decos_Kcs_Email_Entities_Email))
                            mainObject = new SearchObject(loadedData.Id, contentType, loadedData.ObjectSource, loadedData.EmailAddress);
                        else
                            mainObject = new SearchObject(loadedData.Id, contentType, loadedData.ObjectSource, loadedData.ExtendedId);
                        break;
                    }
                case EntityFqdn.Decos_Kcs_ContactHistory_Entities_ContactMoment.toString():
                    {
                        mainObject = new SearchObject(loadedData.Id, contentType, loadedData.ObjectSource);
                        if (loadedData.ObjectDtos && loadedData.ObjectDtos.length > 0) {
                            loadedData.ObjectDtos.forEach(objectData => {
                                let searchObject = new SearchObject(objectData.ObjectId, objectData.ObjectType, objectData.ObjectSource);
                                nativeObjectList.push(searchObject);
                            });
                        }
                        if (loadedData.HandlerDto && !this.isEmpty(loadedData.HandlerDto.UserId))
                            nativeObjectList.push(new SearchObject(loadedData.HandlerDto.UserId, this.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Employees_Entities_Employee]), ""));
                        break;
                    }
                case EntityFqdn.Decos_Kcs_Tasks_Entities_Task.toString():
                    {
                        let mainEntityExists = false;
                        mainObject = new SearchObject(loadedData.Id, 'Decos.Kcs.Tasks.Entities.' + loadedData.TaskType, loadedData.ObjectSource);
                        if (loadedData.RelatedEntityList && loadedData.RelatedEntityList.length > 0) {
                            loadedData.RelatedEntityList.forEach(relatedItem => {
                                let searchObject = new SearchObject(relatedItem.RelatedEntityId, this.toString(EntityFqdn[relatedItem.EntityType]), relatedItem.ObjectSource);
                                if (mainEntityExists == false && relatedItem.RelatedEntityId == loadedData.MainEntityId)
                                    mainEntityExists = true;
                                nativeObjectList.push(searchObject);
                            });
                        }
                        if (loadedData.TaskHistoryList && loadedData.TaskHistoryList.length > 0 && loadedData.TaskHistoryList[0] && loadedData.TaskHistoryList[0].TaskHandlerList && loadedData.TaskHistoryList[0].TaskHandlerList.length > 0) {
                            loadedData.TaskHistoryList[0].TaskHandlerList.forEach(taskHandler => {
                                if (taskHandler && taskHandler.HandlerType == TaskHandlerType[TaskHandlerType.Employee]) {
                                    let searchObject = new SearchObject(taskHandler.HandlerId, this.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Employees_Entities_Employee]), "");
                                    nativeObjectList.push(searchObject);
                                }
                            });
                        }
                        if (loadedData.MainEntityId != null && mainEntityExists == false) {
                            let mainEntityType = EntityFqdn.Decos_Kcs_Persons_Entities_Person;
                            switch (loadedData.MainEntityType) {
                                case 1:
                                    mainEntityType = EntityFqdn.Decos_Kcs_Persons_Entities_Person;
                                    break;
                                case 2:
                                    mainEntityType = EntityFqdn.Decos_Kcs_Organizations_Entities_Organization;
                                    break;
                                case 3:
                                    mainEntityType = EntityFqdn.Decos_Kcs_Notes_Entities_NoteEntity;
                                    break;
                            }
                            let searchObject = new SearchObject(loadedData.MainEntityId, this.toString(EntityFqdn[mainEntityType]), loadedData.MainEntitySource);
                            nativeObjectList.push(searchObject);
                        }
                        break;
                    }
                case EntityFqdn.Decos_Kcs_Tickets_Entities_Ticket.toString():
                    {
                        let mainEntityExists = false;
                        mainObject = new SearchObject(loadedData.Id, contentType, loadedData.ObjectSource);
                        if (loadedData.RelatedEntityList && loadedData.RelatedEntityList.length > 0) {
                            loadedData.RelatedEntityList.forEach(relatedItem => {
                                let searchObject = new SearchObject(relatedItem.EntityId, this.toString(EntityFqdn[relatedItem.EntityType]), relatedItem.ObjectSource);
                                if (mainEntityExists == false && relatedItem.RelatedEntityId == loadedData.MainEntityId)
                                    mainEntityExists = true;
                                nativeObjectList.push(searchObject);
                            });
                        }
                        if (loadedData.ThirdParties && loadedData.ThirdParties.length > 0) {
                            loadedData.ThirdParties.forEach(relatedItem => {
                                let searchObject = new SearchObject(relatedItem.EntityId, this.toString(EntityFqdn[relatedItem.EntityType == 1 ? 9 : 8]), relatedItem.EntitySource);
                                nativeObjectList.push(searchObject);
                            });
                        }
                        if (loadedData.MainEntityId != null && mainEntityExists == false) {
                            let mainEntityType = EntityFqdn.Decos_Kcs_Persons_Entities_Person;
                            switch (loadedData.MainEntityType) {
                                case 1:
                                    mainEntityType = EntityFqdn.Decos_Kcs_Persons_Entities_Person;
                                    break;
                                case 2:
                                    mainEntityType = EntityFqdn.Decos_Kcs_Organizations_Entities_Organization;
                                    break;
                                case 3:
                                    mainEntityType = EntityFqdn.Decos_Kcs_Notes_Entities_NoteEntity;
                                    break;
                            }
                            let searchObject = new SearchObject(loadedData.MainEntityId, this.toString(EntityFqdn[mainEntityType]), loadedData.MainEntitySource);
                            nativeObjectList.push(searchObject);
                        }
                        break;
                    }
                case EntityFqdn.Decos_Kcs_Notes_Entities_InformationResponse.toString():
                case EntityFqdn.Decos_Kcs_Notes_Entities_NoteEntity.toString():
                    {
                        mainObject = new SearchObject(loadedData.Id, contentType, loadedData.ObjectSource);
                        if (loadedData.RelatedEntities && loadedData.RelatedEntities.length > 0) {
                            loadedData.RelatedEntities.forEach(relatedItem => {
                                let searchObject = new SearchObject(relatedItem.RelatedEntityId, relatedItem.ContentType, relatedItem.ObjectSource);
                                nativeObjectList.push(searchObject);
                            });
                        }
                        break;
                    }
                case EntityFqdn.Decos_Kcs_Documents_Entities_Document.toString():
                    {
                        mainObject = new SearchObject(loadedData.Id, contentType, loadedData.ObjectSource, loadedData.CaseId);
                        if (loadedData.Stakeholders && loadedData.Stakeholders.length > 0) {
                            loadedData.Stakeholders.forEach(stakeHolder => {
                                let searchObject = new SearchObject(stakeHolder.StakeholderId, this.convertStakeHolderTypeToContentType(stakeHolder.StakeholderType), "");
                                nativeObjectList.push(searchObject);
                            });
                        }
                        if (loadedData.DocumentObjects && loadedData.DocumentObjects.length > 0) {
                            loadedData.DocumentObjects.forEach(documentObject => {
                                let searchObject = new SearchObject(documentObject.ObjectId, this.convertStakeHolderTypeToContentType(documentObject.ObjectType), "");
                                nativeObjectList.push(searchObject);
                            });
                        }
                        break;
                    }
                case EntityFqdn.Decos_Kcs_Cases_Entities_CaseEntity.toString():
                    {
                        mainObject = new SearchObject(loadedData.Id, contentType, loadedData.ObjectSource);
                        if (loadedData.CaseStakeholders && loadedData.CaseStakeholders.length > 0) {
                            loadedData.CaseStakeholders.forEach(stakeHolder => {
                                if (stakeHolder != null && stakeHolder.PersonStakeholder && stakeHolder.PersonStakeholder != null && stakeHolder.PersonStakeholder.length > 0) {
                                    stakeHolder.PersonStakeholder.forEach(personStakeHolder => {
                                        let searchObject = new SearchObject(personStakeHolder.PersonId, this.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Persons_Entities_Person]), personStakeHolder.ObjectSource);
                                        nativeObjectList.push(searchObject);
                                    });
                                }
                                if (stakeHolder != null && stakeHolder.OrganizationStakeholder && stakeHolder.OrganizationStakeholder != null && stakeHolder.OrganizationStakeholder.length > 0) {
                                    stakeHolder.OrganizationStakeholder.forEach(organizationStakeHolder => {
                                        let searchObject = new SearchObject(organizationStakeHolder.OrganizationId, this.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Organizations_Entities_Organization]), organizationStakeHolder.ObjectSource);
                                        nativeObjectList.push(searchObject);
                                    });
                                }
                                if (stakeHolder != null && stakeHolder.DepartmentStakeholder && stakeHolder.DepartmentStakeholder != null && stakeHolder.DepartmentStakeholder.length > 0) {
                                    stakeHolder.DepartmentStakeholder.forEach(departmentStakeHolder => {
                                        let searchObject = new SearchObject(departmentStakeHolder.DepartmentId, this.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Employees_Entities_Department]), "");
                                        nativeObjectList.push(searchObject);
                                    });
                                }
                                if (stakeHolder != null && stakeHolder.EmployeeStakeholder && stakeHolder.EmployeeStakeholder != null && stakeHolder.EmployeeStakeholder.length > 0) {
                                    stakeHolder.EmployeeStakeholder.forEach(employeeStakeHolder => {
                                        let searchObject = new SearchObject(employeeStakeHolder.EmployeeId, this.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Employees_Entities_Employee]), "");
                                        nativeObjectList.push(searchObject);
                                    });
                                }
                            });
                        }
                        if (loadedData.Petitioner != null) {
                            if (loadedData.Petitioner.PersonStakeholder != null) {
                                let searchObject = new SearchObject(loadedData.Petitioner.PersonStakeholder.PersonId, this.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Persons_Entities_Person]), loadedData.Petitioner.PersonStakeholder.ObjectSource, loadedData.Petitioner.PersonStakeholder.PersonId);
                                nativeObjectList.push(searchObject);
                            }
                            else if (loadedData.Petitioner.OrganizationStakeholder != null) {
                                let searchObject = new SearchObject(loadedData.Petitioner.OrganizationStakeholder.OrganizationId, this.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Organizations_Entities_Organization]), loadedData.Petitioner.OrganizationStakeholder.ObjectSource, loadedData.Petitioner.OrganizationStakeholder.OrganizationId);
                                nativeObjectList.push(searchObject);
                            }
                        }
                        ;
                        if (loadedData.Objects && loadedData.Objects.length > 0) {
                            loadedData.Objects.forEach(caseObject => {
                                if (!this.isEmpty(caseObject.ObjectId)) {
                                    let searchObject = new SearchObject(caseObject.ObjectId, caseObject.ObjectType, caseObject.ObjectSource);
                                    nativeObjectList.push(searchObject);
                                }
                            });
                        }
                        break;
                    }
                case EntityFqdn.Decos_Kcs_CaseTypes_Entities_CaseType.toString():
                    {
                        mainObject = new SearchObject(loadedData.Id, contentType, loadedData.ObjectSource, loadedData.ExtendedId);
                        if (loadedData.CaseTypeDto && loadedData.CaseTypeDto.Department) {
                            let searchObject = new SearchObject(loadedData.CaseTypeDto.Department, this.toString(EntityFqdn[EntityFqdn.Decos_Kcs_Employees_Entities_Employee]), "");
                            nativeObjectList.push(searchObject);
                        }
                        break;
                    }
                case EntityFqdn.Decos_Kcs_SocialMedia_Entities_WhatsApp.toString():
                case EntityFqdn.Decos_Kcs_SocialMedia_Entities_Facebook.toString():
                case EntityFqdn.Decos_Kcs_SocialMedia_Entities_Twitter.toString():
                    {
                        mainObject = new SearchObject(loadedData.Id, contentType, loadedData.ObjectSource, loadedData.ExtendedId);
                        if (loadedData.LinkedEntity != null) {
                            let searchObject = new SearchObject(loadedData.LinkedEntity.TargetId, loadedData.LinkedEntity.TargetType, loadedData.LinkedEntity.TargetSource);
                            nativeObjectList.push(searchObject);
                        }
                        break;
                    }
                case EntityFqdn.Decos_Kcs_Email_Entities_Email.toString():
                    {
                        mainObject = new SearchObject(loadedData.Id, contentType, loadedData.ObjectSource, loadedData.ExtendedId);
                        if (!this.isEmpty(loadedData.LinkedEntityList)) {
                            for (let i = 0; i < loadedData.LinkedEntityList.length; i++) {
                                let searchObject = new SearchObject(loadedData.LinkedEntityList[i].TargetId, loadedData.LinkedEntityList[i].TargetType, loadedData.LinkedEntityList[i].TargetSource);
                                nativeObjectList.push(searchObject);
                            }
                        }
                        break;
                    }
                default:
                    {
                        mainObject = new SearchObject(loadedData.Id, contentType, loadedData.ObjectSource, loadedData.ExtendedId);
                        break;
                    }
            }
        }
        relatedObjectInformation = { mainObject: mainObject, nativeObjectList: nativeObjectList };
        return relatedObjectInformation;
    }
    ;
    static parseContactNumber(contactNumber) {
        let parsedContactNumber = "";
        if (!this.isEmpty(contactNumber)) {
            parsedContactNumber = contactNumber.replace(/-/g, '');
            parsedContactNumber = parsedContactNumber.indexOf('+') > -1 ? parsedContactNumber.substring(3, parsedContactNumber.length) : parsedContactNumber;
            parsedContactNumber = parsedContactNumber.indexOf('0') === 0 ? parsedContactNumber.substring(1, parsedContactNumber.length) : parsedContactNumber;
        }
        return parsedContactNumber;
    }
    ;
    static isStringInList(list, value, isCaseSensitive = false) {
        let retValue = false;
        if (this.isEmpty(list) || this.isEmpty(value))
            return retValue;
        for (let string of list) {
            if (isCaseSensitive && string.toLowerCase() == value.toLowerCase()) {
                retValue = true;
                break;
            }
            else if (!isCaseSensitive && string == value) {
                retValue = true;
                break;
            }
        }
        return retValue;
    }
    static getLastSocialMediaHistoryRecord(list) {
        let result = null;
        if (list != null && list != "undefined" && list.length > 0) {
            for (let i = list.length - 1; i >= 0; i--) {
                if (list[i] != null && list[i].value != null && list[i].value.indexOf("Entities.SocialMedia") != -1) {
                    result = list[i].key;
                    break;
                }
            }
        }
        return result;
    }
    ;
    static clearValue() {
        $(".jkcCircleCloseSearch").hide();
        $(".search-input").val("");
    }
    ;
    static createGuid() {
        return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, (c) => {
            let r = Math.random() * 16 | 0, v = c === 'x' ? r : (r & 0x3 | 0x8);
            return v.toString(16);
        });
    }
    ;
    static getBrowser() {
        let userAgent = navigator.userAgent;
        let browsers = { chrome: /chrome/i, safari: /safari/i, firefox: /firefox/i, ie: /internet explorer/i };
        for (let key in browsers) {
            if (browsers[key].test(userAgent)) {
                return key;
            }
        }
        ;
        return 'unknown';
    }
    ;
    static getLastCaseEntityFromSession(list) {
        let result = null;
        if (list != null && list != "undefined" && list.length > 0) {
            for (let i = list.length - 1; i >= 0; i--) {
                if (list[i] != null && list[i].value != null && list[i].value.indexOf("CaseEntity") != -1) {
                    result = list[i].key;
                    break;
                }
            }
        }
        return result;
    }
    static getCustomerInformationForExternalAppointment(personEntity) {
        let customerInformation = new TaskCustomerInformationDto();
        if (personEntity.Details != null && personEntity.Details.FormalName != null) {
            if (personEntity.Details.FormalName.Gender != null) {
                if (personEntity.Details.FormalName.Gender === 0)
                    customerInformation.Gender = "M";
                else if (personEntity.Details.FormalName.Gender === 1)
                    customerInformation.Gender = "F";
                else if (personEntity.Details.FormalName.Gender === 2)
                    customerInformation.Gender = "U";
            }
            customerInformation.FirstName = personEntity.Details.FormalName.FirstNames[0];
            customerInformation.LastName = personEntity.Details.FormalName.Surname;
            customerInformation.LastNamePrefix = personEntity.Details.FormalName.Prefix;
            customerInformation.DateOfBirth = personEntity.Details.DateOfBirth;
        }
        if (personEntity.Details != null && personEntity.Details.AddressName != null)
            customerInformation.Initials = personEntity.Details.AddressName.Initials;
        if (!this.isEmpty(personEntity.ResidentialAddress) && personEntity.ResidentialAddress.Address != null) {
            customerInformation.StreetName = personEntity.ResidentialAddress.Address.StreetName;
            customerInformation.HouseNumber = personEntity.ResidentialAddress.Address.HouseNumber;
            customerInformation.HouseNumberSuffix = personEntity.ResidentialAddress.Address.HouseNumberAddition;
            customerInformation.PostalCode = personEntity.ResidentialAddress.Address.ZipCode;
            customerInformation.City = personEntity.ResidentialAddress.Address.City;
            customerInformation.CountryCode = ""; // (ISO 2-letterige landcode, zoals NL, BE, DE)
        }
        if (!this.isEmpty(personEntity.EmailAddress))
            customerInformation.EmailAddress = personEntity.EmailAddress;
        if (!this.isEmpty(personEntity.PhoneNumber))
            customerInformation.PhoneNumber = personEntity.PhoneNumber;
        if (!this.isEmpty(personEntity.MobileNumber))
            customerInformation.MobilePhoneNumber = personEntity.MobileNumber;
        return customerInformation;
    }
    static toEntityFqdnString(contentType) {
        let entityFqdn = contentType;
        if (!this.isEmpty(contentType)) {
            switch (contentType.toUpperCase()) {
                case "CASEENTITY":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Cases_Entities_CaseEntity].toString();
                        break;
                    }
                case "CASETYPE":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_CaseTypes_Entities_CaseType].toString();
                        break;
                    }
                case "DOCUMENT":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Documents_Entities_Document].toString();
                        break;
                    }
                case "EMAIL":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Email_Entities_Email].toString();
                        break;
                    }
                case "EMPLOYEE":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Employees_Entities_Employee].toString();
                        break;
                    }
                case "NOTEENTITY":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Notes_Entities_NoteEntity].toString();
                        break;
                    }
                case "ORGANIZATION":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Organizations_Entities_Organization].toString();
                        break;
                    }
                case "PERSON":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Persons_Entities_Person].toString();
                        break;
                    }
                case "PRODUCT":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Products_Entities_Product].toString();
                        break;
                    }
                case "TASK":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Tasks_Entities_Task].toString();
                        break;
                    }
                case "CALLBACKNOTE":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Tasks_Entities_CallbackNote].toString();
                        break;
                    }
                case "ASSIGNMENT":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Tasks_Entities_Assignment].toString();
                        break;
                    }
                case "REPORTIMPROVEMENT":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Tasks_Entities_ReportImprovement].toString();
                        break;
                    }
                case "INFORMATIONREQUEST":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Tasks_Entities_InformationRequest].toString();
                        break;
                    }
                case "INFORMATIONRESPONSE":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Notes_Entities_InformationResponse].toString();
                        break;
                    }
                case "DEPARTMENT":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Employees_Entities_Department].toString();
                        break;
                    }
                case "CONTACTMOMENT":
                case "CONTACTHISTORY":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_ContactHistory_Entities_ContactMoment].toString();
                        break;
                    }
                case "ITEM":
                case "NEWS":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_News_Entities_Item].toString();
                        break;
                    }
                case "QUESTIONANSWER":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Faqs_Entities_QuestionAnswer].toString();
                        break;
                    }
                case "SESSION":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Sessions_Entities_Session].toString();
                        break;
                    }
                case "LOCATION":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Locations_Entities_Building].toString();
                        break;
                    }
                case "FILEENTITY":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Files_Entities_FileEntity].toString();
                        break;
                    }
                case "THEMEPAGE":
                case "THEME":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Themes_Entities_Theme].toString();
                        break;
                    }
                case "LINK":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Links_Entities_Link].toString();
                        break;
                    }
                case "TICKET":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Tickets_Entities_Ticket].toString();
                        break;
                    }
                case "ISSUE":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Issues_Entities_Issue].toString();
                        break;
                    }
                case "DOMAIN":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_Domains_Entities_Domain].toString();
                        break;
                    }
                case "FACEBOOK":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_SocialMedia_Entities_Facebook].toString();
                        break;
                    }
                case "SOCIALMEDIA":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_SocialMedia_Entities_SocialMedia].toString();
                        break;
                    }
                case "WHATSAPP":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_SocialMedia_Entities_WhatsApp].toString();
                        break;
                    }
                case "TWITTER":
                    {
                        entityFqdn = EntityFqdn[EntityFqdn.Decos_Kcs_SocialMedia_Entities_Twitter].toString();
                        break;
                    }
            }
        }
        return entityFqdn;
    }
    ;
    static convertTicketToBase(ticket) {
        let baseDto = new BaseDto();
        baseDto.Id = ticket.Id.toString();
        baseDto.Partial = true;
        baseDto.ObjectSource = "Decos.Kcs.Tickets.Cdr";
        return baseDto;
    }
    static clearTemporaryNote() {
        let temporaryNotesArea = $('#temporaryNotesTextArea');
        if (temporaryNotesArea)
            temporaryNotesArea.val('').empty();
        let temporaryNotesCustomerName = $('#temporaryNotesCustomerName');
        if (temporaryNotesCustomerName)
            temporaryNotesCustomerName.val('').empty();
        let temporaryNotesOrganizationName = $('#temporaryNotesOrganizationName');
        if (temporaryNotesOrganizationName)
            temporaryNotesOrganizationName.val('').empty();
        let temporaryNotesCaseNumber = $('#temporaryNotesCaseNumber');
        if (temporaryNotesCaseNumber)
            temporaryNotesCaseNumber.val('').empty();
        let temporaryNotesCustomerRole = $('#temporaryNotesCustomerRole');
        if (temporaryNotesCustomerRole)
            temporaryNotesCustomerRole.val('0');
    }
    static limitContentOfDropdownWithEllipsis(list, windowWidth) {
        if (list != null && list.length > 0) {
            let maxLength = windowWidth / 8;
            if (list.length > maxLength)
                list = list.substr(0, maxLength) + "...";
        }
        return list;
    }
    ;
    static getScreenResolutionInfo() {
        let resolutionInfo = new ResolutionInfo();
        resolutionInfo.Width = window.innerWidth;
        resolutionInfo.Height = window.innerHeight;
        if (resolutionInfo.Width > 1920)
            resolutionInfo.ResolutionType = ResolutionType.HighResolution;
        else if (resolutionInfo.Width >= 1450 && resolutionInfo.Width <= 1920)
            resolutionInfo.ResolutionType = ResolutionType.MediumResolution;
        else
            resolutionInfo.ResolutionType = ResolutionType.LowerResolution;
        return resolutionInfo;
    }
    ;
    static formatStringValue(formatValue, argumentsList) {
        let s = formatValue;
        for (let i = 0; i < argumentsList.length; i++) {
            let reg = new RegExp("\\{" + i + "\\}", "gm");
            s = s.replace(reg, argumentsList[i]);
        }
        return s;
    }
    ;
    static isPersonConfident(confidentialLevel) {
        return confidentialLevel > 0;
    }
    ;
    static sortMenuItems(menuItemList) {
        if (menuItemList != null && menuItemList.length > 1) {
            let menuItemDetail = menuItemList.splice(0, 1);
            let locationMenuItemIndex = this.findIndex(menuItemList, "contentType", "LOCATION");
            let locationMenuItem = null;
            if (locationMenuItemIndex > -1)
                locationMenuItem = menuItemList.splice(locationMenuItemIndex, 1);
            menuItemList.sort((x, y) => (x.index - y.index));
            menuItemList.splice(0, 0, menuItemDetail[0]);
            if (!this.isEmpty(locationMenuItem))
                menuItemList.splice(1, 0, locationMenuItem[0]);
        }
    }
    static findIndex(arrayList, property, value) {
        let index = -1;
        if (this.isEmpty(arrayList))
            return index;
        for (let iterationIndex = 0; iterationIndex < arrayList.length; iterationIndex++) {
            if (arrayList[iterationIndex][property] == value) {
                index = iterationIndex;
                break;
            }
        }
        return index;
    }
    static capitalize(text) {
        return text.charAt(0).toUpperCase() + text.slice(1);
    }
    static isNumber(val) {
        return /^\d+$/.test(val);
    }
    static checkDashboardRights(contentType) {
        return this.checkRights(contentType) && this.checkRights(contentType, OperationType.ShowDashboardMenu);
    }
    ;
    static getFirstTaskFromSession(list) {
        let result = null;
        if (list != null && list != "undefined" && list.length > 0) {
            for (let i = 0; i < list.length; i++) {
                if (list[i] != null && list[i].value != null && list[i].value.indexOf("Task") != -1) {
                    result = list[i].key;
                    break;
                }
            }
        }
        return result;
    }
    ;
    static getLastEmailFromSession(list) {
        let result = null;
        if (list != null && list != "undefined" && list.length > 0) {
            for (let i = list.length - 1; i >= 0; i--) {
                if (list[i] != null && list[i].value != null && list[i].value.indexOf("Email") != -1) {
                    result = list[i].key;
                    break;
                }
            }
        }
        return result;
    }
    ;
    static getLastLetterFromSession(list) {
        let result = null;
        if (list != null && list != "undefined" && list.length > 0) {
            for (let i = list.length - 1; i >= 0; i--) {
                if (list[i] != null && list[i].value != null && list[i].value.indexOf("Document") != -1 && list[i].key.IsLetter) {
                    result = list[i].key;
                    break;
                }
            }
        }
        return result;
    }
    ;
    static getLastCaseFromSession(list) {
        let result = null;
        if (list != null && list != "undefined" && list.length > 0) {
            for (let i = list.length - 1; i >= 0; i--) {
                if (list[i] != null && list[i].value != null && list[i].value.indexOf("CaseEntity") != -1) {
                    result = list[i].key;
                    break;
                }
            }
        }
        return result;
    }
    ;
    static getLastIssueFromSession(list) {
        let result = null;
        if (list != null && list != "undefined" && list.length > 0) {
            for (let i = list.length - 1; i >= 0; i--) {
                if (list[i] != null && list[i].value != null && list[i].value.indexOf("Issue") != -1) {
                    result = list[i].key;
                    break;
                }
            }
        }
        return result;
    }
    ;
    static getLastDocumentFromSession(list) {
        let result = null;
        if (list != null && list != "undefined" && list.length > 0) {
            for (let i = list.length - 1; i >= 0; i--) {
                if (list[i] != null && list[i].value != null && list[i].value.indexOf("Document") != -1) {
                    result = list[i].key;
                    break;
                }
            }
        }
        return result;
    }
    ;
    static getLastTicketFromSession(list) {
        let result = null;
        if (list != null && list != "undefined" && list.length > 0) {
            for (let i = list.length - 1; i >= 0; i--) {
                if (list[i] != null && list[i].value != null && list[i].value.indexOf("Ticket") != -1) {
                    result = list[i].key;
                    break;
                }
            }
        }
        return result;
    }
    ;
    static getLastSocialMediaFromSession(list) {
        let result = null;
        if (list != null && list != "undefined" && list.length > 0) {
            for (let i = list.length - 1; i >= 0; i--) {
                if (list[i] != null && list[i].value != null && (list[i].value.indexOf("Entities.Facebook") != -1 || list[i].value.indexOf("Entities.Twitter") != -1 || list[i].value.indexOf("Entities.WhatsApp") != -1)) {
                    result = list[i].key;
                    break;
                }
            }
        }
        return result;
    }
    ;
    static isSessionContainsPersonOrganizationRecord(list) {
        let result = false;
        if (list != null && list != "undefined" && list.length > 0) {
            for (let i = 0; i < list.length; i++) {
                if (list[i] != null && list[i].value != null && (list[i].value.indexOf("Person") != -1 || list[i].value.indexOf("Organization") != -1)) {
                    result = true;
                    break;
                }
            }
        }
        return result;
    }
    ;
    static getDateTimeFromTo(dateTime, startTime, endTime) {
        return new this.dateJS(dateTime).toString('dd MMM yyyy') + ' ' + new this.dateJS(startTime).toString('HH:mm') + ' - ' + new this.dateJS(endTime).toString('HH:mm');
    }
    static getFormattedDate(date) {
        let formattedDate;
        if (date.toString().length == 10) {
            let separator = date.toString().charAt(2);
            switch (separator) {
                case '-':
                    formattedDate = new this.dateJS.parse(date, 'dd-MM-yyyy');
                    break;
                case '/':
                    formattedDate = new this.dateJS.parse(date, 'dd/MM/yyyy');
                    break;
                case '.':
                    formattedDate = new this.dateJS.parse(date, 'dd.MM.yyyy');
                    break;
                case ' ':
                    formattedDate = new this.dateJS.parse(date, 'dd MM yyyy');
                    break;
            }
        }
        else if (date.toString().length == 11)
            formattedDate = new Date(date);
        else if (date.toString().length == 8)
            formattedDate = new this.dateJS.parseExact(date, "ddMMyyyy");
        else if (date.toString().length == 6)
            formattedDate = new this.dateJS.parseExact(date, "ddMMyy");
        if (formattedDate && formattedDate.toString() != '[object Object]' && formattedDate.toString() != "Invalid Date")
            formattedDate = new this.dateJS(formattedDate).toString('M/d/yyyy HH:mm:ss');
        return formattedDate;
    }
    ;
    static toString(entityFqdn) {
        let contentType = entityFqdn;
        try {
            //TODO validate whether entityFqdn is numeric or not
            if (!this.isEmpty(entityFqdn)) {
                contentType = entityFqdn.replace(/_/g, '.');
            }
        }
        catch (e) { }
        return contentType;
    }
    ;
    static getHandlerName(id, cacheManager) {
        let handlerName = "";
        if (!this.isEmpty(id)) {
            let employee = cacheManager.getEmployeeById(id);
            if (employee != null && employee != undefined)
                handlerName = employee.StringRepresentation;
        }
        return handlerName;
    }
    ;
    static isPersonDeceased(dateOfExpiration) {
        return !this.isEmpty(dateOfExpiration) && new this.dateJS(dateOfExpiration).toString('M/d/yyyy') != '1/1/1' && dateOfExpiration != '01 Jan 0001';
    }
    static resizejkcBootstrapTable() {
        if ($(".jkcBootstrapTable").length > 0) {
            if ($('#searchPopupDialogSection').length > 0) {
                if ($('#searchPopupDialogSection').height > 0) {
                    let tableHeight = $('#searchPopupDialogSection').height() - 280 + 'px';
                    $("#searchPopupDialog").find(".tableRowsWithLazyLoading").height(tableHeight);
                }
            }
            else {
                let windowHeight = $(window).height();
                let reduceHeight = $('#filterBar').length > 0 ? 256 : 216;
                let newHeight = windowHeight - reduceHeight + "px";
                $(".tableRowsWithLazyLoading").height(newHeight);
                $("table[id$=SearchResultTable]").each(function (key, element) {
                    $("#searchresultoverflowSection").css({ 'overflow': 'auto hidden' });
                });
            }
        }
    }
    ;
    static isValidDiacriticSearch(source, target) {
        return (!this.isEmpty(source) && !this.isEmpty(target) && this.convertDiacriticToPlainText(source).toLowerCase().indexOf(target) >= 0);
    }
    static updateResultListWitTags(isTagFeatureEnabled, result, applicationSettingManager) {
        if (isTagFeatureEnabled) {
            for (let index = 0; index < result.length; index++) {
                result[index].TagList = applicationSettingManager.tagSetting.searchTagById(result[index].Tags);
                let TagString = '';
                result[index].TagList.forEach(tag => {
                    TagString += tag.text + ' ';
                });
                result[index].TagListText = TagString;
            }
        }
    }
    static convertDiacriticToPlainText(text) {
        try {
            if (!this.isEmpty(text)) {
                let retStr = '';
                for (let i = 0; i < text.length; i++) {
                    if (this.diacriticsMap[text[i]])
                        retStr += this.diacriticsMap[text[i]];
                    else
                        retStr += text[i];
                }
                text = retStr;
            }
        }
        catch (e) {
            console.error(e.message);
        }
        return text;
    }
    static getObjectSourceForTask(taskType) {
        let objectSource = null;
        switch (taskType.toUpperCase()) {
            case "CALLBACKNOTE":
                objectSource = TaskDtoSource.CallbackNotes;
                break;
            case "APPOINTMENT":
                objectSource = TaskDtoSource.Appointments;
                break;
            case "ASSIGNMENT":
                objectSource = TaskDtoSource.Assignments;
                break;
            case "INFORMATIONREQUEST":
                objectSource = TaskDtoSource.InformationRequest;
                break;
            case "REPORTIMPROVEMENT":
                objectSource = TaskDtoSource.ReportImprovements;
                break;
        }
        return objectSource;
    }
    static isOperationExcluded(key) {
        let isExcluded = false;
        if (!this.isEmpty(key)) {
            switch (key.toUpperCase()) {
                case "REPLYNOTE":
                case "DELETE ALL":
                case "BACK TO ALL":
                case "TASKFLOW":
                case "CONTACTFLOW":
                    {
                        isExcluded = true;
                        break;
                    }
            }
        }
        return isExcluded;
    }
    static isMobileNumber(number) {
        let mobile = number.toString().replace(/ /g, '');
        mobile = mobile.replace(/-/g, '');
        if (mobile.indexOf("+316") == 0 || mobile.indexOf("316") == 0 || mobile.indexOf("006") == 0 || mobile.indexOf("06") == 0)
            return true;
        else
            return false;
    }
    static isValidToShowMap(privateFields) {
        let isValid = false;
        if (this.isLicenseSpecificFeatureEnabled(LicenseSpecificFeature.AddressesAndBuildings) && this.hasRight(this.toContentType("Location"), OperationType.View)) {
            if ((this.isEmpty(privateFields) || (privateFields.toUpperCase().indexOf("|ADDRESS|") === -1 && privateFields.toUpperCase().indexOf("|ZIPCODE|") === -1 && privateFields.toUpperCase().indexOf("|STREETNAME|") === -1 && privateFields.toUpperCase().indexOf("|HOUSENUMBER|") === -1)))
                isValid = true;
        }
        return isValid;
    }
    ;
    static getFqdnList() {
        let fqdnList = [];
        fqdnList.push("PERSON");
        fqdnList.push("ORGANIZATION");
        fqdnList.push("CASEENTITY");
        fqdnList.push("LOCATION");
        fqdnList.push("DOCUMENT");
        fqdnList.push("FILEDOCUMENT");
        fqdnList.push("CASETYPE");
        fqdnList.push("PRODUCT");
        fqdnList.push("NOTEENTITY");
        fqdnList.push("SCRIPT");
        fqdnList.push("CONTACTMOMENT");
        fqdnList.push("TASK");
        fqdnList.push("EMAIL");
        fqdnList.push("EMPLOYEE");
        fqdnList.push("QUESTIONANSWER");
        return fqdnList;
    }
    static getLastOrganizationFromSession(list) {
        let result = null;
        if (list != null && list != "undefined" && list.length > 0) {
            for (let i = list.length - 1; i >= 0; i--) {
                if (list[i] != null && list[i].value != null && list[i].value.indexOf("Organization") != -1) {
                    result = list[i].key;
                    break;
                }
            }
        }
        return result;
    }
    static removeItem(items, item) {
        if (item != null && items != null && items.length > 0) {
            for (let i = 0; i < items.length; i++) {
                if (items[i].Id == item.Id) {
                    items.splice(i, 1);
                    break;
                }
            }
        }
        return items;
    }
    ;
    static autoCompleteSearchBoxOperations(item, event, productService, sessionManager) {
        let isBtnClicked = false;
        try {
            if (event != null && event.target != null && event.target != null && event.target.id != null) {
                if (event.target.id.toUpperCase().indexOf("EFORMBTN") >= 0) {
                    isBtnClicked = true;
                    this.stopEventPropagation(event);
                    if (productService != null && item != null && item.Links != null && item.Links.LinkDto != null && !this.isEmpty(item.Links.LinkDto.UriText))
                        productService.showEform(item.Links.LinkDto.UriText);
                }
                else if (event.target.id.toUpperCase().indexOf("CURRENTLOCATIONBTN") >= 0) {
                    isBtnClicked = true;
                    this.stopEventPropagation(event);
                    sessionManager.geoCurrentLocation();
                    $('#typeheadSearchBox').hide();
                }
            }
        }
        catch (e) {
            console.error(e);
        }
        return isBtnClicked;
    }
    ;
    static isSessionContainsEmailRecord(list) {
        let result = false;
        if (list != null && list != "undefined" && list.length > 0) {
            for (let i = 0; i < list.length; i++) {
                if (list[i] != null && list[i].value != null && (list[i].value.indexOf("Email") != -1)) {
                    result = true;
                    break;
                }
            }
        }
        return result;
    }
    ;
    static getFirstEmailFromSession(list) {
        let result = null;
        if (list != null && list != "undefined" && list.length > 0) {
            for (let i = 0; i < list.length; i++) {
                if (list[i] != null && list[i].value != null && list[i].value.indexOf("Email") != -1) {
                    result = list[i].key;
                    break;
                }
            }
        }
        return result;
    }
    ;
    static isValidPhonenumber(phonenumber, countryCode) {
        let isValid = false;
        if (!this.isEmpty(phonenumber)) {
            phonenumber = phonenumber.replace(/\s/g, ''); // trim spaces
            if (phonenumber.indexOf('00') == 0)
                phonenumber = phonenumber.substr(2);
            if (phonenumber.indexOf('+') == 0) // '+' only allowed at start 
                phonenumber = phonenumber.substr(1);
            let repititionRegex = /([-\(\)])\1+/; // "-, ) and (" should not be repeated [ ex: ((, --, )) ] 
            let endsWithDigitRegex = /\d$/; // should end with a digit
            if (!repititionRegex.test(phonenumber) && endsWithDigitRegex.test(phonenumber)) {
                // now its safe to remove the characters  - ) and ( from phone number, only 10 digits should left out.
                phonenumber = phonenumber.replace(/[\(\)-]/g, '');
                let length = phonenumber.length;
                //banco di carbie and OLB
                if (countryCode == "+599") {
                    if (length >= 7 && length <= 11 && this.isNumber(phonenumber))
                        isValid = true;
                }
                else {
                    if (length >= 10 && length <= 14 && this.isNumber(phonenumber))
                        isValid = true;
                }
            }
        }
        return isValid;
    }
    ;
    static remove_duplicates(objectsArray) {
        let usedObjects = {};
        for (let i = objectsArray.length - 1; i >= 0; i--) {
            let so = JSON.stringify(objectsArray[i]);
            if (usedObjects[so]) {
                objectsArray.splice(i, 1);
            }
            else {
                usedObjects[so] = true;
            }
        }
    }
    static getCaseNumberFromSession(list) {
        let result = null;
        if (list != null && list != "undefined" && list.length > 0) {
            for (let i = list.length - 1; i >= 0; i--) {
                if (list[i] != null && list[i].value != null && list[i].value.indexOf("CaseEntity") != -1) {
                    result = list[i].key;
                    break;
                }
            }
        }
        if (result != null)
            return result.Identification;
        else
            return "";
    }
    static ensureHttpPrefix(value) {
        if (value && !/^(https?):\/\//i.test(value) && 'http://'.indexOf(value) !== 0 && 'https://'.indexOf(value) !== 0) {
            return 'http://' + value;
        }
        else
            return value;
    }
    static popUpCenter(url, objWidth, objHeight) {
        try {
            let dualScreenLeft = window.screenLeft != undefined ? window.screenLeft : screen["left"];
            let dualScreenTop = window.screenTop != undefined ? window.screenTop : screen["top"];
            let width = window.innerWidth ? window.innerWidth : document.documentElement.clientWidth ? document.documentElement.clientWidth : screen.width;
            let height = window.innerHeight ? window.innerHeight : document.documentElement.clientHeight ? document.documentElement.clientHeight : screen.height;
            let left = ((width / 2) - (objWidth / 2)) + dualScreenLeft;
            let top = ((height / 2) - (objHeight / 2)) + dualScreenTop;
            window.open(url, "_blank", "toolbar=0,location=0,menubar=0,status=1,scrollbars=1,resizable=1,width=" + objWidth + ",height=" + objHeight + ", top=" + top + ", left=" + left);
        }
        catch (e) {
            return false;
        }
    }
    static isContainsPersonOrganizationName(selectedEntity) {
        let result = '';
        if (selectedEntity && this.isEmpty(selectedEntity.MainEntityId)) {
            if (selectedEntity.RelatedEntityFqdn.includes('Ticket'))
                result = selectedEntity.CustomerName;
            else if (selectedEntity.RelatedEntityFqdn.includes('CallbackNote'))
                result = selectedEntity.CustomerOrOrganizationName;
        }
        return result;
    }
    static setPlaceholder(id, placeholderText) {
        let inputbox = $(id).find("input");
        if (inputbox && typeof inputbox[0] !== 'undefined')
            $('#' + inputbox[0].id).attr('placeholder', placeholderText);
    }
    static getGenderString(gend) {
        let gender = 'Unknown';
        switch (gend) {
            case 0:
                gender = 'Male';
                break;
            case 1:
                gender = 'Female';
                break;
        }
        return gender;
    }
    static setEmployeeExtraProperties(employeeDto) {
        if (employeeDto != null) {
            let jsonHandlerString = JSON.stringify(employeeDto);
            let currentHandlerJsonType = JsonDtoType.Employee;
            jsonHandlerString = jsonHandlerString.substr(1);
            jsonHandlerString = currentHandlerJsonType + jsonHandlerString;
            employeeDto = JSON.parse(jsonHandlerString);
        }
        return employeeDto;
    }
    ;
    static validDate(value) {
        let isValid = false;
        if (Object.prototype.toString.call(value) === "[object Date]") {
            // it is a date
            if (isNaN(value.getTime())) { // d.valueOf() could also work
                // date is not valid
                isValid = false;
            }
            else {
                // date is valid
                isValid = true;
            }
        }
        return isValid;
    }
    ;
    static removeByAttr(arr, attr, value) {
        let i = arr.length;
        while (i--) {
            if (arr[i] && arr[i].hasOwnProperty(attr) && (arguments.length > 2 && arr[i][attr] === value))
                arr.splice(i, 1);
        }
        return arr;
    }
    static getTaskHandlerNameList(taskHandlerDtoList, cacheManager, separtor = ", ") {
        let handlerList = "";
        if (handlerList != null && !this.isEmpty(taskHandlerDtoList) && taskHandlerDtoList.length > 0) {
            for (let index1 = 0; index1 < taskHandlerDtoList.length; index1++) {
                if (taskHandlerDtoList[index1].HandlerType == TaskHandlerType.Employee || taskHandlerDtoList[index1].HandlerType == "Employee") {
                    let employee = cacheManager.getEmployeeById(taskHandlerDtoList[index1].HandlerId);
                    if (employee != null && employee != undefined) {
                        if (handlerList == "")
                            handlerList = employee.StringRepresentation;
                        else
                            handlerList += separtor + employee.StringRepresentation;
                    }
                }
                else if (taskHandlerDtoList[index1].HandlerType == TaskHandlerType.Department || taskHandlerDtoList[index1].HandlerType == "Department") {
                    let department = cacheManager.getDepartmentById(taskHandlerDtoList[index1].HandlerId);
                    if (department != null && department != undefined) {
                        if (handlerList == "")
                            handlerList = department.Name;
                        else
                            handlerList += separtor + department.Name;
                    }
                }
                else if (taskHandlerDtoList[index1].HandlerType == TaskHandlerType.Mailbox || taskHandlerDtoList[index1].HandlerType == "Mailbox") {
                    let taskmailbox = cacheManager.getTaskMailboxById(taskHandlerDtoList[index1].HandlerId);
                    if (taskmailbox) {
                        if (handlerList == "")
                            handlerList = taskmailbox.Name;
                        else
                            handlerList += separtor + taskmailbox.Name;
                    }
                }
                else if (taskHandlerDtoList[index1].HandlerType == TaskHandlerType.Email || taskHandlerDtoList[index1].HandlerType == "Email") {
                    if (handlerList == "")
                        handlerList = taskHandlerDtoList[index1].HandlerId;
                    else
                        handlerList += separtor + taskHandlerDtoList[index1].HandlerId;
                }
            }
        }
        return handlerList;
    }
    static toBoolean(text) {
        switch (text.toLowerCase().trim()) {
            case "true":
            case "yes":
            case "1": return true;
            case "false":
            case "no":
            case "0":
            case null: return false;
            default: return Boolean(text);
        }
    }
    ;
    static highlightTab(message, favIcon, appTitle, sessionManager, phoneState, chatService) {
        let windowBlinkTimeoutId;
        let appFavIcon = "assets/images/favicon.ico";
        let checkPickupInterval;
        let checkHangupInterval;
        let checkAccpetInterval;
        let checkIgnoredInterval;
        let blink = function () {
            if (document.title == message) {
                document.title = appTitle;
                $("link[rel='icon']").attr("href", appFavIcon);
            }
            else {
                document.title = message;
                $("link[rel='icon']").attr("href", favIcon);
            }
        };
        let blinkInterval = setInterval(blink, 1000);
        if (phoneState) {
            let checkCallState = function (callState) {
                if (!GlobalFunctions.isEmpty(phoneState) && phoneState.CurrentState == callState) {
                    clearInterval(blinkInterval);
                    document.title = appTitle;
                    $("link[rel='icon']").attr("href", appFavIcon);
                }
            };
            checkPickupInterval = setInterval(() => checkCallState(CallStatus.Active), 1000);
            checkHangupInterval = setInterval(() => checkCallState(CallStatus.Inactive), 1000);
        }
        if (chatService) {
            let checkchatstatus = function (ChatState) {
                if (!GlobalFunctions.isEmpty(chatService) && chatService.CurrentStatus == ChatState) {
                    clearInterval(blinkInterval);
                    document.title = appTitle;
                    $("link[rel='icon']").attr("href", appFavIcon);
                }
            };
            checkAccpetInterval = setInterval(() => checkchatstatus(ChatState.Accepted), 1000);
            checkIgnoredInterval = setInterval(() => checkchatstatus(ChatState.Ignored), 1000);
        }
        windowBlinkTimeoutId = () => {
            clearInterval(blinkInterval);
            clearInterval(checkPickupInterval);
            clearInterval(checkHangupInterval);
            clearInterval(checkAccpetInterval);
            clearInterval(checkIgnoredInterval);
            document.title = appTitle;
            $("link[rel='icon']").attr("href", appFavIcon);
        };
        return windowBlinkTimeoutId;
    }
    static getPriorityText(priority, translateService) {
        let priorityText = "";
        switch (priority) {
            case 1:
                {
                    priorityText = translateService.instant("LBL_LOW");
                    break;
                }
            case 2:
                {
                    priorityText = translateService.instant("LBL_MEDIUM");
                    break;
                }
            case 3:
                {
                    priorityText = translateService.instant("LBL_HIGH");
                    break;
                }
        }
        return priorityText;
    }
    ;
    static getPublishText(publishTo, localize) {
        let publishText = "";
        if (!this.isEmpty(publishTo)) {
            switch (publishTo) {
                case 0:
                    {
                        publishText = localize.instant("LBL_MYSELF");
                        break;
                    }
                case 1:
                    {
                        publishText = localize.instant("LBL_MY_DEPARTMENT");
                        break;
                    }
                case 2:
                    {
                        publishText = localize.instant("LBL_EVERYONE");
                        break;
                    }
            }
        }
        else
            publishText = localize.instant("LBL_MYSELF");
        return publishText;
    }
    ;
    static fetchEmailSource(emailId) {
        let emailSource = "";
        if (!GlobalFunctions.isEmpty(emailId)) {
            let emailArray = emailId.split("|");
            if (emailArray && emailArray.length == 2) {
                emailSource = EmailSource[emailArray[1]];
            }
        }
        return emailSource;
    }
    ;
    static sortDictionary(unorderedObj) {
        let orderedObj = {};
        Object.keys(unorderedObj).sort().forEach(function (key) {
            orderedObj[key] = unorderedObj[key];
        });
        return orderedObj;
    }
    static getLastPersonOrOrganizationEmailAddress(detailViewCachedEntityList) {
        let emailAddress = "";
        if (!GlobalFunctions.isEmpty(detailViewCachedEntityList)) {
            for (let i = detailViewCachedEntityList.length - 1; i >= 0; i--) {
                if (detailViewCachedEntityList[i] != null && detailViewCachedEntityList[i].value != null && (detailViewCachedEntityList[i].value.indexOf("Person") != -1 || detailViewCachedEntityList[i].value.indexOf("Organization") != -1)) {
                    emailAddress = detailViewCachedEntityList[i].key.EmailAddress;
                    break;
                }
            }
        }
        return emailAddress;
    }
    static getBasketName(basketItem) {
        let basketName = "";
        if (!GlobalFunctions.isEmpty(basketItem)) {
            if (!GlobalFunctions.isEmpty(basketItem.Item) && !GlobalFunctions.isEmpty(basketItem.Item.Name))
                basketName = basketItem.Item.Name;
            else if (!GlobalFunctions.isEmpty(basketItem.JsonItem)) {
                let jsonItem = JSON.parse(basketItem.JsonItem);
                basketName = jsonItem.StringRepresentation;
            }
            if (GlobalFunctions.isEmpty(basketName))
                basketName = EntityFqdn[basketItem.Type].toString();
        }
        return basketName;
    }
    ;
    static clearAppCacheData() {
        this.userPrivilegeData = null;
        this.licenseSpecificFeatureList = [];
    }
    static supportsDeeplinking(type, source) {
        if (!GlobalFunctions.isEmpty(type) && !GlobalFunctions.isEmpty(source))
            return ['appointment', 'assignment', 'callbacknote', 'informationrequest', 'reportimprovement', 'ticket'].includes(type) && source.toLowerCase().includes('cdr');
        else
            return false;
    }
    static editorIsEmpty(editor) {
        if (GlobalFunctions.isEmpty(editor)) {
            $(".tox-edit-area").addClass('error');
            return true;
        }
        else {
            $(".tox-edit-area").removeClass('error');
            return false;
        }
    }
    static addWorkDays(n) {
        let date = new Date();
        let num = Math.abs(n);
        let tem, count = 0;
        let dir = (n < 0) ? -1 : 1;
        while (count < num) {
            date = new Date(date.setDate(date.getDate() + dir));
            //if (date.isHoliday()) continue;
            tem = date.getDay();
            if (tem != 0 && tem != 6)
                ++count;
        }
        return date;
    }
    static placeCall(phoneManager, phoneNumber, name, picture, id = null) {
        if (phoneManager.phoneState.IsPhoneFeatureEnabled)
            phoneManager.placeCall(phoneNumber, name, picture, id);
    }
    static loadSendEmailDialog(emailId, dialogManager) {
        if (GlobalFunctions.checkRights('Email') && !GlobalFunctions.isEmpty(emailId))
            dialogManager.loadSendEmailDialog(emailId);
    }
    static navigateWithOptionalQueryParam(router, route, replaceUrl = false) {
        const routeParts = route?.split('?');
        if (!routeParts[1])
            router.navigate([routeParts[0]]);
        else {
            const keyValueParts = routeParts[1].split('=');
            if (!keyValueParts[2])
                router.navigate([routeParts[0]], { queryParams: { [keyValueParts[0]]: keyValueParts[1] }, replaceUrl });
            else if (route?.indexOf('list/email') > -1) {
                const emailParams = routeParts[1].split('&');
                if (emailParams.length === 2) {
                    const filterParam = emailParams[0].split('=');
                    const mailboxParam = emailParams[1].split('=');
                    router.navigate([routeParts[0]], { queryParams: { [filterParam[0]]: filterParam[1], [mailboxParam[0]]: mailboxParam[1] }, replaceUrl });
                }
            }
        }
    }
    static refreshCalendar(calendarApi, calendarOptions, calendarView = 'timeGridWeek', agendaItems, currentDate, currentAvailability, calendarContentHeight, translateService, selectedSlotCallback, allowSelection, slotLength, hideFullDay, isSelectedSlotDisplay = false, dateTimeNow = null) {
        try {
            if (calendarApi) {
                calendarApi.removeAllEvents();
                let isAnyAgendaWithFullDayFound = false;
                if (!allowSelection && !hideFullDay) {
                    calendarApi.addEvent({
                        allDay: false,
                        editable: false,
                        backgroundColor: '#34b7db',
                        end: currentDate.setHours(17, 0, 0),
                        start: currentDate.setHours(8, 0, 0)
                    });
                }
                if (currentAvailability.length > 0) {
                    const events = [];
                    for (let slot = 0; slot < currentAvailability.length; slot++) {
                        let agendaFound = false;
                        if (agendaItems?.length > 0) {
                            for (let i = 0; i < agendaItems.length; i++) {
                                if (agendaItems[i].StartDate <= currentAvailability[slot].StartDate && agendaItems[i].EndDate >= currentAvailability[slot].EndDate) {
                                    agendaFound = true;
                                    break;
                                }
                            }
                        }
                        if (!agendaFound && currentAvailability[slot].FreeBusyStatus != FreeBusyStatus.Free) {
                            const event = {
                                title: '',
                                allDay: false,
                                editable: false,
                                borderColor: '#FFD4D4',
                                backgroundColor: '#FFD4D4',
                                end: currentAvailability[slot].EndDate,
                                start: currentAvailability[slot].StartDate
                            };
                            // Merge consecutive 15 minute slots into one big slot instead of keeping them scattered for better overview
                            events.length > 0 && events[events.length - 1].end.valueOf() === event.start.valueOf() ? (events[events.length - 1].end = event.end) : events.push(event);
                        }
                    }
                    events.forEach(event => calendarApi.addEvent(event));
                }
                if (agendaItems.length > 0) {
                    for (let i = 0; i < agendaItems.length; i++) {
                        if (agendaItems[i].IsAllDay)
                            isAnyAgendaWithFullDayFound = true;
                        calendarApi.addEvent({
                            editable: false,
                            textColor: 'black',
                            end: agendaItems[i].EndDate,
                            title: agendaItems[i].Subject,
                            start: agendaItems[i].StartDate,
                            allDay: agendaItems[i].IsAllDay,
                            FreeBusyStatus: agendaItems[i].FreeBusyStatus,
                            backgroundColor: this.isEmpty(agendaItems[i].CategoryColor) ? '#CDE6F7' : agendaItems[i].CategoryColor
                        });
                    }
                }
                calendarOptions.contentHeight = calendarContentHeight;
                calendarOptions.dayHeaders = calendarView === 'timeGridWeek';
                calendarOptions.allDaySlot = isAnyAgendaWithFullDayFound;
                calendarOptions.selectable = allowSelection;
                calendarOptions.selectMirror = allowSelection;
                calendarOptions.editable = allowSelection;
                calendarOptions.select = selectedSlotCallback;
                calendarOptions.weekends = calendarView === 'timeGridWeek' ? false : true;
                calendarOptions.slotDuration = slotLength === '120' ? '02:00:00' : '00:' + slotLength + ':00';
                calendarApi.gotoDate(currentDate);
                calendarApi.changeView(calendarView);
                calendarOptions.eventDidMount = ({ el, event }) => {
                    let agendaTitle = event.title + '\n\n' + translateService.instant("LBL_START") + ': ' + new this.dateJS(event.start).toString('dd MMM yyyy HH:mm') + '\n' + translateService.instant("LBL_END") + ': ' + new this.dateJS(event.end).toString('dd MMM yyyy HH:mm');
                    if (!this.isEmpty(event.extendedProps.FreeBusyStatus)) {
                        switch (event.extendedProps.FreeBusyStatus) {
                            case AgendaShowAsStatus.Busy:
                                el.classList.add("agendaShowAsStatusBusy");
                                agendaTitle += '\n' + translateService.instant("LBL_SHOW_AS") + ': ' + translateService.instant("LBL_NOT_AVAILABLE");
                                break;
                            case AgendaShowAsStatus.Free:
                                el.classList.add("agendaShowAsStatusFree");
                                agendaTitle += '\n' + translateService.instant("LBL_SHOW_AS") + ': ' + translateService.instant("LBL_AVAILABLE");
                                break;
                            case AgendaShowAsStatus.WorkingElsewhere:
                                el.classList.add("agendaShowAsStatusTentativeOrWorkingElsewhere");
                                agendaTitle += '\n' + translateService.instant("LBL_SHOW_AS") + ': ' + translateService.instant("LBL_WORKING_ELSEWHERE");
                                break;
                            case AgendaShowAsStatus.Tentative:
                                el.classList.add("agendaShowAsStatusTentativeOrWorkingElsewhere");
                                agendaTitle += '\n' + translateService.instant("LBL_SHOW_AS") + ': ' + translateService.instant("LBL_TENTATIVE");
                                break;
                            case AgendaShowAsStatus.OOF:
                                el.classList.add("agendaShowAsStatusOutOfOffice");
                                agendaTitle += '\n' + translateService.instant("LBL_SHOW_AS") + ': ' + translateService.instant("LBL_OUT_OF_OFFICE");
                                break;
                        }
                    }
                    el.setAttribute('title', agendaTitle);
                };
            }
        }
        catch (e) {
            console.error(e);
        }
    }
    static updateCalendarWeekView(enableWorkweek, userSettingManager, systemComponentService) {
        userSettingManager.setUserSettingValueByKey("CalendarWeekView", enableWorkweek.toString());
        systemComponentService.updateUserSettingForCalendarWeekView(enableWorkweek.toString());
    }
    ;
    static isIE() {
        let myNav = navigator.userAgent.toLowerCase();
        return myNav.indexOf('msie') !== -1;
    }
    static handleMissingImage(event) {
        event.target.src = 'assets/images/nophoto.png';
    }
    static getFileUploadErrorText(error, translateService) {
        let fileUploadErrorTex = translateService.instant("LBL_FILE_UPLOAD_FAILED");
        if (!GlobalFunctions.isEmpty(error)) {
            switch (error.status) {
                case 406:
                    {
                        fileUploadErrorTex = translateService.instant("LBL_ERROR_MSG_FILETYPE");
                        ;
                        break;
                    }
                case 413:
                    {
                        fileUploadErrorTex = translateService.instant("LBL_ERROR_MSG_FILE_SIZE");
                        break;
                    }
            }
        }
        return fileUploadErrorTex;
    }
    ;
    static clearInterval() {
        clearInterval(this.timer);
    }
    static getFullCalendarPlugins() {
        return [
            listPlugin,
            momentPlugin,
            dayGridPlugin,
            timeGridPlugin,
            interactionPlugin
        ];
    }
    static toggleDropdownById(id) {
        const elementClassList = document.getElementById(id).classList;
        elementClassList.contains('show') ? elementClassList.remove('show') : elementClassList.add('show');
    }
    static isSameDetailPath(oldPath, newPath) {
        const oldParts = oldPath?.split('/');
        const newParts = newPath?.split('/');
        return (oldParts[0] === newParts[0]) && (oldParts[1] === newParts[1]);
    }
    static formatDaysWithLeadingZeros() {
        // handle internal html representation of day with leading zero(1-9 digits) for automation
        const dateCells = document.querySelectorAll('.custom-cell .mat-calendar-body-cell-content');
        dateCells.forEach(cell => {
            if (parseInt(cell.innerHTML) < 10) {
                cell.classList.add('leading-zero');
                cell.innerHTML = `0${parseInt(cell.innerHTML)}`;
            }
        });
    }
    static removeDotAfterMonth(dateString) {
        // Remove the dot after any month
        return dateString?.replace(/(\s)([a-zA-Z])(\w+)\./g, (match, before, firstLetter, rest) => {
            return before + firstLetter + rest;
        });
    }
    static removeHTMLTags(inputString) {
        if (/<[^>]+>/i.test(inputString)) {
            var plainText = inputString.replace(/<[^>]+>/g, '');
            return plainText;
        }
        else
            return inputString;
    }
    static detectHtmlInjection(text) {
        const htmlRegex = /<\s*\/?[a-z][^>]*>|&[a-z]+;/ig;
        return htmlRegex.test(text);
    }
    static parseNonEnglishDate(dateString) {
        const months = {
            'jan': 'jan',
            'feb': 'feb',
            'mrt': 'mar',
            'apr': 'apr',
            'mei': 'may',
            'jun': 'jun',
            'jul': 'jul',
            'aug': 'aug',
            'sep': 'sep',
            'okt': 'oct',
            'nov': 'nov',
            'dec': 'dec',
            'januari': 'jan',
            'februari': 'feb',
            'maart': 'mar',
            'april': 'apr',
            'juni': 'jun',
            'juli': 'jul',
            'augustus': 'aug',
            'september': 'sep',
            'oktober': 'oct',
            'november': 'nov',
            'december': 'dec',
        };
        const parts = dateString.trim().split(" ");
        const monthName = months.hasOwnProperty(parts[1]?.toLowerCase());
        if (parts?.length === 3 && months[parts[1]?.toLowerCase()] && monthName) {
            parts[1] = months[parts[1]?.toLowerCase()];
            const englishDateString = parts.join(' ');
            return englishDateString;
        }
        else
            return dateString.trim();
    }
    static getServerSideCookie(name) {
        const value = `; ${document.cookie}`;
        const parts = value.split(`; ${name}=`);
        if (parts.length === 2) {
            let cookieValue = parts.pop().split(';').shift();
            this.isEmpty(cookieValue);
            return !this.isEmpty(cookieValue) ? cookieValue : null;
        }
    }
    ;
    static removeDeeplinkLocalStorage() {
        localStorage.removeItem(LocalStorageKey.REDIRECT_URL);
        localStorage.removeItem(LocalStorageKey.DEEPLINK);
    }
    ;
    static downloadFileAsBlob(response) {
        if (!this.isEmpty(response)) {
            let fileName = response.headers.get('content-disposition')?.split(';')[1].split('=')[1];
            fileName = fileName.trim().replace(/"/g, '');
            let blob = response.body;
            let downloadLink = document.createElement('a');
            downloadLink.download = fileName;
            downloadLink.href = window.URL.createObjectURL(blob);
            downloadLink.click();
        }
    }
    ;
    static handleManualDateInput(target, sessionManager) {
        if (!GlobalFunctions.isEmpty(target)) {
            let dateFormat;
            let formattedDate;
            if (target.value.length === 8)
                dateFormat = sessionManager.defaultOptions.ddMMyyyy;
            else if (target.value.length === 6)
                dateFormat = sessionManager.defaultOptions.ddMMyy;
            else if (target.value.length === 10)
                dateFormat = sessionManager.defaultOptions.dateMonthYear;
            else if (target.value.length === 11)
                dateFormat = sessionManager.defaultOptions.defaultDateFormat;
            if (!GlobalFunctions.isEmpty(dateFormat))
                formattedDate = parse(target.value, dateFormat, new Date());
            return formattedDate;
        }
    }
    static setHandlerListFromTaskTemplate(currentTemplate, cacheManager, handlerTags) {
        let currentHandlerType = "Employee";
        currentTemplate.HandlerIds.forEach(handlerId => {
            let handleData = null;
            if (currentTemplate.HandlerType == TaskHandlerType.Employee.toString())
                handleData = cacheManager.getEmployeeById(handlerId);
            else if (currentTemplate.HandlerType == TaskHandlerType.Department.toString()) {
                handleData = cacheManager.getDepartmentById(handlerId);
                currentHandlerType = "Department";
            }
            else if (currentTemplate.HandlerType == TaskHandlerType.Mailbox.toString()) {
                handleData = cacheManager.getTaskMailboxById(handlerId);
                currentHandlerType = "Mailbox";
            }
            if (handleData != null) {
                handleData.text = handleData.StringRepresentation;
                handlerTags.push(handleData);
            }
        });
        return currentHandlerType;
    }
    static setRequiredFieldsFromTaskTemplate(currentTemplate) {
        let fieldsObj = new RequiredFields();
        currentTemplate.RequiredFields.forEach(field => {
            switch (field) {
                case "contactperson": {
                    fieldsObj.isContactPersonRequired = true;
                    break;
                }
                case "emailaddress": {
                    fieldsObj.isEmailRequired = true;
                    break;
                }
                case "address": {
                    fieldsObj.isAddressRequired = true;
                    break;
                }
                case "phonenumber": {
                    fieldsObj.isPhoneNumberRequired = true;
                    break;
                }
            }
        });
        return fieldsObj;
    }
    static encodeHtml(input) {
        //encode the unsafe code
        const entityMap = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#39;',
        };
        return input.replace(/[&<>"']/g, (match) => entityMap[match]);
    }
    static convertStringDate(dateString, dateFormat, datePipe) {
        if (!dateString)
            return null;
        const trimmedDate = dateString.trim();
        const parsedDate = moment(trimmedDate, 'YYYY-MM-DDTHH:mm:ss').toDate();
        return datePipe.transform(parsedDate, dateFormat);
    }
    static isEmailDomainAllowed(emailAddress, domains) {
        let isAlllowed = true;
        if (!this.isEmpty(emailAddress) && !this.isEmpty(domains)) {
            let domainList = domains.toLowerCase().split(';');
            for (var index = 0; index < domainList.length; index++) {
                if (emailAddress.toLowerCase().indexOf('@' + domainList[index]) != -1) {
                    isAlllowed = false;
                    break;
                }
            }
        }
        return isAlllowed;
    }
    static isTaskDomain(domain) {
        return `${Domain.TASK.ALL}|${Domain.TASK.APPOINTMENT}|${Domain.TASK.ASSIGNMENT}|${Domain.TASK.CALLBACK_NOTE}|${Domain.TASK.INFORMATION_REQUEST}|${Domain.TASK.REPORT_IMPROVEMENT}`.includes(domain);
    }
    static isValidDate(dateString) {
        let date = new Date(dateString);
        return !isNaN(date.getTime());
    }
    static tryParseToDateString(query, datePipe, defaultDateFormat) {
        let d1;
        let retVal = query;
        try {
            if (query.length == 8 && !isNaN(query) && GlobalFunctions.isNumber(query))
                d1 = new Date(query.substring(4), query.substring(2, 4) - 1, query.substring(0, 2));
            else {
                if (/^\d{1,2}[-\/]\d{1,2}[-\/]\d{4}$/.test(query)) { // test for a complete date query string
                    let delimiter = "";
                    let dateParts = query.split("-");
                    if (dateParts.length == 3 || dateParts.length == 2)
                        delimiter = "-";
                    else {
                        dateParts = query.split("/");
                        if (dateParts.length == 3)
                            delimiter = "/";
                        else {
                            dateParts = query.split(".");
                            if (dateParts.length == 3)
                                delimiter = ".";
                            else
                                dateParts = query.split(" ");
                        }
                    }
                    if (dateParts.length == 3)
                        d1 = new Date(dateParts[2], dateParts[1] - 1, dateParts[0]);
                }
            }
            if (d1 && !isNaN(d1.getTime()))
                retVal = datePipe.transform(d1, defaultDateFormat);
        }
        catch (e) {
            retVal = query;
        }
        return retVal;
    }
    static compressImage(blob, maxWidth, maxHeight, quality) {
        return new Promise((resolve, reject) => {
            const reader = new FileReader();
            reader.onload = event => {
                const img = new Image();
                img.onload = () => {
                    let { width, height } = img;
                    if (width > maxWidth) {
                        height *= maxWidth / width;
                        width = maxWidth;
                    }
                    if (height > maxHeight) {
                        width *= maxHeight / height;
                        height = maxHeight;
                    }
                    const canvas = document.createElement('canvas');
                    canvas.width = width;
                    canvas.height = height;
                    const ctx = canvas.getContext('2d');
                    ctx.drawImage(img, 0, 0, width, height);
                    canvas.toBlob(blob => blob ? resolve(blob) : reject(new Error('Compression failed')), 'image/jpeg', quality);
                };
                img.onerror = reject;
                img.src = event.target.result;
            };
            reader.onerror = reject;
            reader.readAsDataURL(blob);
        });
    }
    static sanitizeAndTruncate(text, maxLength = 200) {
        // Replace newlines and excessive whitespace
        let cleaned = text.replace(/\s+/g, ' ').trim();
        // Truncate to a safe length
        if (cleaned.length > maxLength)
            cleaned = cleaned.substring(0, maxLength) + '...';
        return cleaned;
    }
    static setEditorBodyHeight() {
        let iframe = document.querySelector('iframe.tox-edit-area__iframe');
        let wrapper = document.querySelector('.tox-editor-container');
        if (iframe && iframe.contentDocument && iframe.contentDocument.body && wrapper) {
            let containerHeight = wrapper.clientHeight;
            iframe.contentDocument.body.style.height = `${containerHeight}px`;
        }
    }
    static openSummarizationInNewWindow(content, title, url) {
        let payload = { title, content };
        let popupWidth = 1024;
        let popupHeight = 550;
        let left = (window.screen.width - popupWidth) / 2;
        let top = (window.screen.height - popupHeight) / 2;
        let newWindow = window.open(url, '_blank', `width=${popupWidth},height=${popupHeight},top=${top},left=${left},scrollbars=1,resizable=1`);
        newWindow?.addEventListener('load', () => {
            newWindow.postMessage(payload, '*');
            newWindow.opener = null;
        });
    }
}
GlobalFunctions.dateJS = Date;
GlobalFunctions.userPrivilegeData = null;
GlobalFunctions.licenseSpecificFeatureList = [];
GlobalFunctions.templateUniqueId = GlobalFunctions.getGuid();
GlobalFunctions.diacriticsMap = {
    '\u00C0': 'A',
    '\u00C1': 'A',
    '\u00C2': 'A',
    '\u00C3': 'A',
    '\u00C4': 'A',
    '\u00C5': 'A',
    '\u00C6': 'AE',
    '\u00C7': 'C',
    '\u00C8': 'E',
    '\u00C9': 'E',
    '\u00CA': 'E',
    '\u00CB': 'E',
    '\u00CC': 'I',
    '\u00CD': 'I',
    '\u00CE': 'I',
    '\u00CF': 'I',
    '\u0132': 'IJ',
    '\u00D0': 'D',
    '\u00D1': 'N',
    '\u00D2': 'O',
    '\u00D3': 'O',
    '\u00D4': 'O',
    '\u00D5': 'O',
    '\u00D6': 'O',
    '\u00D8': 'O',
    '\u0152': 'OE',
    '\u00DE': 'TH',
    '\u00D9': 'U',
    '\u00DA': 'U',
    '\u00DB': 'U',
    '\u00DC': 'U',
    '\u00DD': 'Y',
    '\u0178': 'Y',
    '\u00E0': 'a',
    '\u00E1': 'a',
    '\u00E2': 'a',
    '\u00E3': 'a',
    '\u00E4': 'a',
    '\u00E5': 'a',
    '\u00E6': 'ae',
    '\u00E7': 'c',
    '\u00E8': 'e',
    '\u00E9': 'e',
    '\u00EA': 'e',
    '\u00EB': 'e',
    '\u00EC': 'i',
    '\u00ED': 'i',
    '\u00EE': 'i',
    '\u00EF': 'i',
    '\u0133': 'ij',
    '\u00F0': 'd',
    '\u00F1': 'n',
    '\u00F2': 'o',
    '\u00F3': 'o',
    '\u00F4': 'o',
    '\u00F5': 'o',
    '\u00F6': 'o',
    '\u00F8': 'o',
    '\u0153': 'oe',
    '\u00DF': 'ss',
    '\u00FE': 'th',
    '\u00F9': 'u',
    '\u00FA': 'u',
    '\u00FB': 'u',
    '\u00FC': 'u',
    '\u00FD': 'y',
    '\u00FF': 'y',
    '\uFB00': 'ff',
    '\uFB01': 'fi',
    '\uFB02': 'fl',
    '\uFB03': 'ffi',
    '\uFB04': 'ffl',
    '\uFB05': 'ft',
    '\uFB06': 'st' // ﬆ => st
};
/**
 *
 * @returns returns DateTime
 */
GlobalFunctions.getCurrentDateAndTime = () => {
    let DateJS = Date;
    return new DateJS().toString(DateTimeFormat.FORMAT_1);
};
GlobalFunctions.getIconClass = (contentType) => {
    let iconStyleClass = '';
    switch (contentType) {
        case ContentType.Case:
            {
                iconStyleClass = 'jkcCaseBlue jkcSizeSmall';
                break;
            }
        case ContentType.CallbackNote:
            {
                iconStyleClass = 'jkcCallbacknoteBlue jkcSizeSmall';
                break;
            }
        case ContentType.ContactMoment:
            {
                iconStyleClass = 'jkcContacthistoryBlue jkcSizeSmall';
                break;
            }
        case ContentType.Document:
            {
                iconStyleClass = 'jkcDocumentBlue jkcSizeSmall';
                break;
            }
        case ContentType.Email:
            {
                iconStyleClass = 'jkcEmailBlue jkcSizeNormal';
                break;
            }
        case ContentType.Issue:
            {
                iconStyleClass = 'jkcIssueBlue jkcSizeSmall';
                break;
            }
        case ContentType.Employee:
            {
                iconStyleClass = 'jkcEmployeeBlue jkcSizeSmall';
                break;
            }
        case ContentType.File:
            {
                iconStyleClass = 'jkcDocumentBlue jkcSizeSmall';
                break;
            }
        case ContentType.Note:
            {
                iconStyleClass = 'jkcNoteentityBlue jkcSizeSmall';
                break;
            }
        case ContentType.Organization:
            {
                iconStyleClass = 'jkcOrganizationBlue jkcSizeSmall';
                break;
            }
        case ContentType.Person:
            {
                iconStyleClass = 'jkcPersonBlue jkcSizeSmall';
                break;
            }
        case ContentType.Product:
            {
                iconStyleClass = 'jkcProductBlue jkcSizeSmall';
                break;
            }
        case ContentType.Task:
            {
                iconStyleClass = 'jkcTaskBlue jkcSizeSmall';
                break;
            }
        case ContentType.QuestionAnswer:
            {
                iconStyleClass = 'jkcQuestionanswerBlue jkcSizeSmall';
                break;
            }
        case ContentType.Theme:
            {
                iconStyleClass = 'jkcThemeBlue jkcSizeSmall';
                break;
            }
        case ContentType.Link:
            {
                iconStyleClass = 'jkcLinkBlue jkcSizeSmall';
                break;
            }
        case ContentType.News:
            {
                iconStyleClass = 'jkcNewsBlue jkcSizeSmall';
                break;
            }
        case ContentType.Ticket:
            {
                iconStyleClass = 'jkcTicketBlue jkcSizeSmall';
                break;
            }
        case ContentType.Domain:
            {
                iconStyleClass = 'jkcDomainBlue jkcSizeSmall';
                break;
            }
    }
    return iconStyleClass;
};
GlobalFunctions.getEntityFqdnFromSourceFqdn = function (sourceFqdn) {
    let entityName = sourceFqdn.split('.')[2];
    let entityFqdn;
    switch (entityName.toUpperCase()) {
        case "CASES":
            {
                entityFqdn = EntityFqdn.Decos_Kcs_Cases_Entities_CaseEntity;
                break;
            }
        case "PHONE":
            {
                entityFqdn = EntityFqdn.Decos_Kcs_Phone_Entities_Phone;
                break;
            }
        case "BASKETS":
        case "EMAIL":
            {
                entityFqdn = EntityFqdn.Decos_Kcs_Email_Entities_Email;
                break;
            }
        case "PRODUCTS":
            {
                entityFqdn = EntityFqdn.Decos_Kcs_Products_Entities_Product;
                break;
            }
        case "TASKS":
            {
                entityFqdn = EntityFqdn.Decos_Kcs_Tasks_Entities_Task;
                break;
            }
        case "SOCIALMEDIA":
            {
                switch (sourceFqdn.toUpperCase()) {
                    case "DECOS.KCS.SOCIALMEDIA.FACEBOOK":
                        {
                            entityFqdn = EntityFqdn.Decos_Kcs_SocialMedia_Entities_Facebook;
                            break;
                        }
                    case "DECOS.KCS.SOCIALMEDIA.TWITTER":
                        {
                            entityFqdn = EntityFqdn.Decos_Kcs_SocialMedia_Entities_Twitter;
                            break;
                        }
                    case "DECOS.KCS.SOCIALMEDIA.WHATSAPP":
                        {
                            entityFqdn = EntityFqdn.Decos_Kcs_SocialMedia_Entities_WhatsApp;
                            break;
                        }
                }
                break;
            }
        case "CONTACTHISTORY":
            {
                entityFqdn = EntityFqdn.Decos_Kcs_ContactHistory_Entities_ContactHistory;
                break;
            }
        case "AVAILABILITY":
        case "EMPLOYEES":
            {
                entityFqdn = EntityFqdn.Decos_Kcs_Employees_Entities_Employee;
                break;
            }
        case "FILES":
            {
                entityFqdn = EntityFqdn.Decos_Kcs_Files_Entities_FileEntity;
                break;
            }
        case "THEMES":
            {
                entityFqdn = EntityFqdn.Decos_Kcs_Themes_Entities_Theme;
                break;
            }
        case "NOTES":
            {
                entityFqdn = EntityFqdn.Decos_Kcs_Notes_Entities_NoteEntity;
                break;
            }
        case "PERSONS":
            {
                entityFqdn = EntityFqdn.Decos_Kcs_Persons_Entities_Person;
                break;
            }
        case "DOCUMENTS":
            {
                entityFqdn = EntityFqdn.Decos_Kcs_Documents_Entities_Document;
                break;
            }
        case "TICKETS":
            {
                entityFqdn = EntityFqdn.Decos_Kcs_Tickets_Entities_Ticket;
                break;
            }
        case "CHAT":
            {
                entityFqdn = EntityFqdn.Decos_Kcs_Chat_Entities_Chat;
                break;
            }
        case "REPORTS":
            {
                entityFqdn = EntityFqdn.Decos_Kcs_Reports_Entities_Report;
                break;
            }
        case "ISSUES":
            {
                entityFqdn = EntityFqdn.Decos_Kcs_Issues_Entities_Issue;
                break;
            }
        case "DOMAINS":
            {
                entityFqdn = EntityFqdn.Decos_Kcs_Domains_Entities_Domain;
                break;
            }
    }
    return entityFqdn;
};
GlobalFunctions.getValueByPropertyName = function (propertyName, source) {
    let result = null;
    if (source != null && !this.isEmpty(propertyName))
        result = source[propertyName];
    return result;
};
GlobalFunctions.getContentTypeByPropertyName = function (propertyName, source) {
    let result = this.getValueByPropertyName(propertyName, source);
    if (result != null)
        result = this.toString(result);
    return result;
};
GlobalFunctions.isOwnerOfTask = function (taskDto) {
    let isOwnerOfTask = false;
    if (taskDto != null && taskDto.TaskHistoryList != null && taskDto.TaskHistoryList.length > 0 && taskDto.TaskHistoryList[0].TaskHandlerList != null) {
        for (let item in taskDto.TaskHistoryList[0].TaskHandlerList) {
            if (taskDto.TaskHistoryList[0].TaskHandlerList[item].HandlerType == "Mailbox") {
                isOwnerOfTask = true;
                break;
            }
            else if (taskDto.TaskHistoryList[0].TaskHandlerList[item].HandlerType == "Employee" && taskDto.TaskHistoryList[0].TaskHandlerList[item].HandlerId != null) {
                if (taskDto.TaskHistoryList[0].TaskHandlerList[item].HandlerId == this.getLoggedInUserId()) {
                    isOwnerOfTask = true;
                    break;
                }
            }
        }
    }
    return isOwnerOfTask;
};
GlobalFunctions.stopEventPropagation = (event) => {
    if (event.stopPropagation)
        event.stopPropagation();
    if (event.preventDefault)
        event.preventDefault();
    event.cancelBubble = true;
    event.returnValue = false;
};
GlobalFunctions.getLastEmployeeFromSession = list => {
    let result = null;
    if (list != null && list != "undefined" && list.length > 0) {
        for (let i = list.length - 1; i >= 0; i--) {
            if (list[i] != null && list[i].value != null && list[i].value.indexOf("Employee") != -1) {
                result = list[i].key;
                break;
            }
        }
    }
    return result;
};
GlobalFunctions.getRelatedViewsIndex = (contentType) => {
    if (contentType == "CONTACTMOMENT")
        return 0;
    else if (contentType == "TASK")
        return 1;
    else
        return 2;
};
//# sourceMappingURL=global-functions.js.map