import { HttpHeaders } from '@angular/common/http';
import { BroadCastEventKeys, LanaguageCode, Locale, LocalStorageKey } from '../../shared/constants';
import { GlobalFunctions } from '../../shared/global-functions';
import { ExceptionHelper } from '../../shared/helper/exception-helper';
export class BaseService {
    constructor(zone, router, httpClient, eventProvider) {
        this.zone = zone;
        this.router = router;
        this.httpClient = httpClient;
        this.eventProvider = eventProvider;
        this.roleId = '';
        this.isLicensed = true;
        this.isConnected = false;
        this.isConnecting = false;
        this.isReconnecting = false;
        this.isServiceStoppedDialogShown = false;
    }
    disconnect() {
        this.connection = null;
        this.isConnected = false;
        this.isConnecting = false;
        this.isReconnecting = false;
    }
    getHeader() {
        return new HttpHeaders().append(LocalStorageKey.SESSION_TOKEN, GlobalFunctions.getSessionTokenAsString());
    }
    sessionExpired(translateService, dialog) {
        if (!this.eventProvider.isSessionExpired) {
            dialog.closeAll();
            this.eventProvider.isSessionExpired = true;
            ExceptionHelper.showException(dialog, ExceptionHelper.create(translateService.instant('LBL_SESSION_EXPIRED'), translateService.instant('LBL_SESSION_EXPIRED_MSG'), translateService.instant('LBL_SESSION')), true, null, null, this.zone).afterClosed().subscribe(() => this.eventProvider.BroadcastEvent(BroadCastEventKeys.LOGOUT));
        }
    }
    serviceStopped(title, translateService, dialog) {
        if (!this.isServiceStoppedDialogShown && !this.eventProvider.isSessionExpired) {
            dialog.closeAll();
            this.isServiceStoppedDialogShown = true;
            ExceptionHelper.showException(dialog, null, true, translateService.instant('LBL_MSG_SERVICE_UNAVAILABLE'), title, this.zone);
        }
    }
    setSessionToken(service, translateService, dialog) {
        return new Promise(resolve => {
            if (service) {
                if (this.isValidToken()) {
                    if (service.isConnected)
                        return resolve(true);
                    else
                        service.connect().then(isValid => resolve(isValid));
                }
                else {
                    this.sessionExpired(translateService, dialog);
                    return resolve(false);
                }
            }
        });
    }
    // Stabilize JKC by only sending the requests from setSessionToken after service is connected 
    waitForConnection(resolve) {
        const interval = setInterval(() => {
            if (!this.isConnecting) {
                clearInterval(interval);
                return resolve(this.isConnected);
            }
        }, 1000);
    }
    isValidToken() {
        return JSON.parse(GlobalFunctions.getCookie(LocalStorageKey.SESSION_TOKEN)) ?? null;
    }
    getLanguageCode() {
        return navigator.language === LanaguageCode.EN_CODE ? Locale.EN_EN : Locale.NL_NL;
    }
    getData(url) {
        return this.httpClient.get(url, { headers: this.getHeader() });
    }
}
//# sourceMappingURL=base-service.js.map